﻿using System;
using System.Windows.Forms;
using System.Drawing;
using System.Drawing.Imaging;
using System.Threading;

namespace LCDImageUploader
{
    public partial class Form1 : Form
    {
        const int IMAGE_WIDTH   = 128;
        const int IMAGE_HEIGHT  = 160;
        const int SERIAL_BAUD   = 115200;
  
        private serial serialPort;
        private image lcdImage          = new image(IMAGE_WIDTH, IMAGE_HEIGHT);
        private bool uploading          = false;
        private bool editImage          = false;

        public Form1()
        {
            InitializeComponent();

            this.Icon           = System.Drawing.Icon.ExtractAssociatedIcon(System.Reflection.Assembly.GetExecutingAssembly().Location);
            this.Text           = "LCD Image Uploader v" + System.Reflection.Assembly.GetExecutingAssembly().GetName().Version.ToString();
            this.AllowDrop      = true;
            this.DragEnter      += new DragEventHandler(Form1_DragEnter);
            this.DragDrop       += new DragEventHandler(Form1_DragDrop);
            this.FormClosing    += new FormClosingEventHandler(Form1_Closing);

            this.picPreview.MouseDown   += new MouseEventHandler(picPreview_MouseDown);
            this.picPreview.MouseUp     += new MouseEventHandler(picPreview_MouseUp);
            this.picPreview.MouseMove   += new MouseEventHandler(picPreview_MouseMove);
            this.picPreview.MouseEnter  += new EventHandler(picPreview_MouseEnter);
            this.picPreview.MouseLeave  += new EventHandler(picPreview_MouseLeave);
            this.picPreview.MouseWheel  += new MouseEventHandler(picPreview_MouseWheel);
        }

        private void Form1_Load(object sender, EventArgs e)
        {
            serialPort = new serial();
            updatePorts();

            lcdImage.setPreviewBox(picPreview);

            // Set open file dialog filter to only show supported image formats
            openFileDialog1.Filter = "";
            ImageCodecInfo[] codecs = ImageCodecInfo.GetImageEncoders();
            String sep = String.Empty;
            String allExtensions = String.Empty;
            foreach(ImageCodecInfo c in codecs)
            {
                String codecName = c.CodecName.Substring(8).Replace("Codec", "Files").Trim();
                String extensions = c.FilenameExtension.ToLower();
                openFileDialog1.Filter = String.Format("{0}{1}{2} ({3})|{3}", openFileDialog1.Filter, sep, codecName, extensions);
                allExtensions += extensions + ";";
                sep = "|";
            }
            allExtensions = allExtensions.TrimEnd(';');
            openFileDialog1.Filter = String.Format("{0}|{1}|{2}", "Image files", allExtensions, openFileDialog1.Filter);
            openFileDialog1.Filter = String.Format("{0}{1}{2} ({3})|{3}", openFileDialog1.Filter, sep, "All Files", "*.*");
        }

        // Close form
        private void Form1_Closing(object sender, System.ComponentModel.CancelEventArgs e)
        {
            serialPort.close();
        }

        // Drag and drop
        void Form1_DragEnter(object sender, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
                e.Effect = DragDropEffects.Copy;
        }

        // Drag and drop
        void Form1_DragDrop(object sender, DragEventArgs e)
        {
            string[] files = (string[])e.Data.GetData(DataFormats.FileDrop);
            loadImage(files[0]);
        }

        // Mouse enter preview image (for mouse wheel)
        void picPreview_MouseEnter(object sender, EventArgs e)
        {
            picPreview.Focus();
            this.Cursor = Cursors.SizeAll;
        }

        // Mouse leave preview image (for mouse wheel)
        void picPreview_MouseLeave(object sender, EventArgs e)
        {
            lblPreview.Focus();
            this.Cursor = Cursors.Default;
        }

        // Mouse wheel change preview image
        void picPreview_MouseWheel(object sender, MouseEventArgs e)
        {
            if (!uploading)
            {
                float amount = 0;
                if (e.Delta > 0)
                    amount = 0.1F;
                else if (e.Delta < 0)
                    amount = -0.1F;
                lcdImage.doZoom(amount);
            }
        }

        // Mouse down on preview image (for mouse move)
        void picPreview_MouseDown(object sender, MouseEventArgs e)
        {
            editImage = true;
        }

        // Mouse up on preview image (for mouse move)
        void picPreview_MouseUp(object sender, MouseEventArgs e)
        {
            editImage = false;
        }

        // Mouse move on preview image
        void picPreview_MouseMove(object sender, MouseEventArgs e)
        {
            if (editImage && !uploading)
            {
                if (e.Button == MouseButtons.Left)
                {
                    Point mousePos = Cursor.Position;
                    Point picPos = picPreview.PointToScreen(Point.Empty);
                    lcdImage.move(mousePos.X - picPos.X, mousePos.Y - picPos.Y);
                }
                else if (e.Button == MouseButtons.Right)
                    lcdImage.rotate(e.X);
            }
        }

        // Refresh ports button
        private void btnRefreshPorts_Click(object sender, EventArgs e)
        {
            updatePorts();
        }

        // Open image button
        private void btnOpenImg_Click(object sender, EventArgs e)
        {
            openFileDialog1.ShowDialog();
        }

        // Upload button
        private void btnUpload_Click(object sender, EventArgs e)
        {
            uploadImage();
        }

        // Fill drop down list with available ports
        private void updatePorts()
        {
            // Get ports
            string[] ports = serialPort.getPorts();

            // Clear list
            cbPorts.Items.Clear();
            cbPorts.Text = "";

            // Add ports to list
            foreach (string port in ports)
                cbPorts.Items.Add(port);

            // If a port is available then select it
            if (cbPorts.Items.Count > 0)
                cbPorts.SelectedIndex = 0;
        }

        // Open selected port
        private bool openPort(ref string errorStr)
        {
            // No ports listed
            if (cbPorts.Items.Count <= 0)
            {
                errorStr = "No ports available";
                return false;
            }

            // Get selected text
            string port = cbPorts.SelectedItem.ToString();
            if (port.Length < 4) // must be COM###
            {
                errorStr = "Invalid port";
                return false;
            }

            // Try to open port
            if (!serialPort.open(port, SERIAL_BAUD))
            {
                errorStr = serialPort.lastError();
                return false;
            }

            return true;
        }

        // Image selected from open file dialog
        private void openFileDialog1_FileOk(object sender, System.ComponentModel.CancelEventArgs e)
        {
            loadImage(openFileDialog1.FileName);
        }
        
        // Load image
        private void loadImage(string file)
        {
            if(!uploading)
            {
                if (!lcdImage.load(file))
                    MessageBox.Show("Error opening image: " + lcdImage.lastError(), "Image error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }
        
        // Try opening port then upload data
        private void uploadImage()
        {
            if (!uploading)
            {
                // Try opening port
                string errorStr = "";
                if (!openPort(ref errorStr))
                {
                    serialPort.close();
                    MessageBox.Show(errorStr, "Serial error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }

                // Begin uploading
                MethodInvoker mi = new MethodInvoker(uploadImageInvoke);
                mi.BeginInvoke(null, null);
            }
        }

        // Asynchronous data upload
        private void uploadImageInvoke()
        {
            uploading = true;

            // Get colour mode
            bool _12bit = rb12bit.Checked;
            bool _16bit = rb16bit.Checked;
            bool _18bit = rb18bit.Checked;

            // Small delay so controller resets download state
            Thread.Sleep(100);

            // Buffer each row, multiply bufferSize for colour mode
            int bufferSize = IMAGE_WIDTH;
            if (_12bit)
                bufferSize = (int)(bufferSize * 1.5);
            else if (_16bit)
                bufferSize *= 2;
            else if (_18bit)
                bufferSize *= 3;
            byte[] buffer = new byte[bufferSize];
            int bufferIdx = 0;

            //DateTime start = DateTime.Now;

            // Get each pixel colour
            for (int y = IMAGE_HEIGHT - 1; y >= 0; y--)
            {
                for (int x = IMAGE_WIDTH - 1; x >= 0; x--)
                {
                    if (_18bit)
                        buffer18bit(ref buffer, ref bufferIdx, x, y);
                    else if (_16bit)
                        buffer16bit(ref buffer, ref bufferIdx, x, y);
                    else
                        buffer12bit(ref buffer, ref bufferIdx, x, y);
                }

                // Update progress bar
                pbUpload.BeginInvoke(new progressDelegate(updateProgressBar), (object)(int)(100 * ((double)(IMAGE_HEIGHT - y) / (double)IMAGE_HEIGHT)));

                // Serial send
                if (!serialPort.send(buffer, buffer.Length))
                {
                    MessageBox.Show(serialPort.lastError(), "Serial error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    break;
                }

                // Reset buffer pointer
                bufferIdx = 0;
            }

            //DateTime end = DateTime.Now;
            //MessageBox.Show("Time: " + (end.Subtract(start).TotalMilliseconds) + "ms");

            pbUpload.BeginInvoke(new progressDelegate(updateProgressBar), (object)0);
            serialPort.close();
            uploading = false;
        }

        // Buffer 18 bit pixel
        private void buffer18bit(ref byte[] buffer, ref int bufferIdx, int x, int y)
        {
            int colour = lcdImage.GetPixel18(x, y);
            buffer[bufferIdx++] = (byte)(colour >> 16);
            buffer[bufferIdx++] = (byte)(colour >> 8);
            buffer[bufferIdx++] = (byte)colour;
        }

        // Buffer 16 bit pixel
        private void buffer16bit(ref byte[] buffer, ref int bufferIdx, int x, int y)
        {
            short colour = lcdImage.GetPixel16(x, y);
            buffer[bufferIdx++] = (byte)(colour >> 8);
            buffer[bufferIdx++] = (byte)colour;
        }

        // Buffer 12 bit pixel
        private void buffer12bit(ref byte[] buffer, ref int bufferIdx, int x, int y)
        {
            short colour = lcdImage.GetPixel12(x, y);
            if (x % 2 == 1)
            {
                buffer[bufferIdx++] = (byte)(colour >> 4);
                buffer[bufferIdx++] = (byte)(colour >> 8);
            }
            else
            {
                buffer[bufferIdx - 1] |= (byte)(colour << 4);
                buffer[bufferIdx++] = (byte)colour;
            }
        }

        // Update progress bar
        private delegate void progressDelegate(int val);
        private void updateProgressBar(int val)
        {
            pbUpload.Value = val;
        }
    }
}
