/*
 * Project: AVR ATtiny CPU Usage LEDs
 * Author: Zak Kemble, me@zakkemble.co.uk
 * Copyright: (C) 2012 by Zak Kemble
 * License: GNU GPL v3 (see License.txt)
 */

// Source: http://stackoverflow.com however I can't find the original post
// Modified into a C++ class and better support for multiple processors

#include "cpu.h"

cpu::cpu()
{
	SYSTEM_INFO sysinfo;
	GetSystemInfo(&sysinfo);
	cpuCount = sysinfo.dwNumberOfProcessors;

	ulinitVal = new ULONG[(cpuCount + 1) * 2]; //+1 for the total
	ulVal = new ULONG[(cpuCount + 1) * 2]; //+1 for the total

	pLoc = NULL;
	pSvc = NULL;
}

cpu::~cpu()
{
//	delete ulinitVal;
//	delete ulVal;
    pSvc->Release();
    pLoc->Release();
    CoUninitialize();
}

bool cpu::cpuInit()
{
    // Step 1: --------------------------------------------------
    // Initialize COM. ------------------------------------------
    HRESULT hres =  CoInitializeEx(0, COINIT_MULTITHREADED); 
    if (FAILED(hres))
    {
        cout << "Failed to initialize COM library. Error code = 0x" << hex << hres << endl;
        return false;
    }

    // Step 2: --------------------------------------------------
    // Set general COM security levels --------------------------
    // Note: If you are using Windows 2000, you need to specify -
    // the default authentication credentials for a user by using
    // a SOLE_AUTHENTICATION_LIST structure in the pAuthList ----
    // parameter of CoInitializeSecurity ------------------------

    hres = CoInitializeSecurity(
        NULL, 
        -1,                          // COM authentication
        NULL,                        // Authentication services
        NULL,                        // Reserved
        RPC_C_AUTHN_LEVEL_DEFAULT,   // Default authentication 
        RPC_C_IMP_LEVEL_IMPERSONATE, // Default Impersonation  
        NULL,                        // Authentication info
        EOAC_NONE,                   // Additional capabilities 
        NULL                         // Reserved
        );

    if(FAILED(hres))
    {
        cout << "Failed to initialize security. Error code = 0x" << hex << hres << endl;
        CoUninitialize();
        return false;
    }

    // Step 3: ---------------------------------------------------
    // Obtain the initial locator to WMI -------------------------

    hres = CoCreateInstance(
        CLSID_WbemLocator,             
        0, 
        CLSCTX_INPROC_SERVER, 
        IID_IWbemLocator, (LPVOID *) &pLoc);

    if(FAILED(hres))
    {
        cout << "Failed to create IWbemLocator object." << " Err code = 0x" << hex << hres << endl;
        CoUninitialize();
        return false;
    }

    // Step 4: -----------------------------------------------------
    // Connect to WMI through the IWbemLocator::ConnectServer method

    // Connect to the root\cimv2 namespace with
    // the current user and obtain pointer pSvc
    // to make IWbemServices calls.
    hres = pLoc->ConnectServer(
         _bstr_t(L"ROOT\\CIMV2"), // Object path of WMI namespace
         NULL,                    // User name. NULL = current user
         NULL,                    // User password. NULL = current
         0,                       // Locale. NULL indicates current
         NULL,                    // Security flags.
         0,                       // Authority (e.g. Kerberos)
         0,                       // Context object 
         &pSvc                    // pointer to IWbemServices proxy
         );

    if(FAILED(hres))
    {
        cout << "Could not connect. Error code = 0x" << hex << hres << endl;
        pLoc->Release();     
        CoUninitialize();
        return false;
    }

    cout << "Connected to ROOT\\CIMV2 WMI namespace" << endl;

    // Step 5: --------------------------------------------------
    // Set security levels on the proxy -------------------------

    hres = CoSetProxyBlanket(
       pSvc,                        // Indicates the proxy to set
       RPC_C_AUTHN_WINNT,           // RPC_C_AUTHN_xxx
       RPC_C_AUTHZ_NONE,            // RPC_C_AUTHZ_xxx
       NULL,                        // Server principal name 
       RPC_C_AUTHN_LEVEL_CALL,      // RPC_C_AUTHN_LEVEL_xxx 
       RPC_C_IMP_LEVEL_IMPERSONATE, // RPC_C_IMP_LEVEL_xxx
       NULL,                        // client identity
       EOAC_NONE                    // proxy capabilities 
    );

    if(FAILED(hres))
    {
        cout << "Could not set proxy blanket. Error code = 0x" << hex << hres << endl;
        pSvc->Release();
        pLoc->Release();     
        CoUninitialize();
        return false;
    }

	return true;
}

bool cpu::GetValues(bool setInitVals)
{
	// Step 6: --------------------------------------------------
    // Use the IWbemServices pointer to make requests of WMI ----

    // For example, get the name of the operating system
    IEnumWbemClassObject* pEnumerator = NULL;
    HRESULT hres = pSvc->ExecQuery(
        bstr_t("WQL"), 
        bstr_t("SELECT * FROM Win32_PerfRawData_PerfOS_Processor"),
        WBEM_FLAG_FORWARD_ONLY | WBEM_FLAG_RETURN_IMMEDIATELY, 
        NULL,
        &pEnumerator);

    if(FAILED(hres))
    {
        cout << "Query for processor perf data failed." << " Error code = 0x" << hex << hres << endl;
		pEnumerator->Release();
        return false;
    }

    // Step 7: -------------------------------------------------
    // Get the data from the query in step 6 -------------------

    IWbemClassObject *pclsObj = NULL;
    ULONG uReturn = 0;
	int nCtr = 0;

	ULONG* tmp = NULL;
	tmp = setInitVals ? ulinitVal : ulVal;

	int n = (cpuCount + 1) * 2;

	VARIANT* vtProp = new VARIANT;

    while (nCtr < n)
    {
        hres = pEnumerator->Next(WBEM_INFINITE, 1, &pclsObj, &uReturn);
		
        if(!uReturn)
            break;

        VariantInit(vtProp);
		hres = pclsObj->Get(L"PercentProcessorTime", 0, vtProp, 0, 0);
		swscanf_s(vtProp->bstrVal, L"%I64u", &tmp[nCtr]);
        VariantClear(vtProp);

        VariantInit(vtProp);
		hres = pclsObj->Get(L"TimeStamp_Sys100NS", 0, vtProp, 0, 0);
		swscanf_s(vtProp->bstrVal, L"%I64u", &tmp[nCtr + 1]);
		VariantClear(vtProp);
		
		pclsObj->Release();
		pclsObj = NULL;
		
		nCtr += 2;
    }
	vtProp = NULL;
	delete vtProp;  // remove this line for XP... fixes crash but then creates a memory leak :/ I think the line above (vtProp = NULL;) fixed this problem
	if(pclsObj != NULL)
		pclsObj->Release();
	pEnumerator->Release();
	return true;
}

double cpu::calcUsage(int core)
{
	if(core == -1) // Total all cores
		core = ((cpuCount + 1) * 2) - 2;

	double timestampDiff = ulVal[core + 1]- ulinitVal[core + 1];
	if(timestampDiff <= 0) // dont div by 0!
		return 0;
	double cpuTimeDiff = ulVal[core] - ulinitVal[core];
	return fabs(100.0 - (cpuTimeDiff / timestampDiff) * 100);
}
