/*
 * Project: AVR ATtiny CPU Usage LEDs
 * Author: Zak Kemble, me@zakkemble.co.uk
 * Copyright: (C) 2012 by Zak Kemble
 * License: GNU GPL v3 (see License.txt)
 */

#include "mainBackend.h"

HWND* settingsWnd;
HWND* trackbar;

usb* USB = NULL;
cpu* CPU = NULL;

extern bool settingsDlgOpen;

DWORD WINAPI mainThread(LPVOID args);

void backend(HWND* _settingsWnd, HWND* _trackbar)
{
	settingsWnd		= _settingsWnd;
	trackbar		= _trackbar;
	DWORD threadId;
	HANDLE hThread	= CreateThread(NULL, 0, mainThread, NULL, 0, &threadId);
}

void setDefaults()
{

}

// Put into single colour mode and set the colour
void setSingleColour()
{
	unsigned int r = GetDlgItemInt(*settingsWnd, IDC_EDIT_SGL_RED, NULL, false);
	unsigned int g = GetDlgItemInt(*settingsWnd, IDC_EDIT_SGL_GREEN, NULL, false);
	unsigned int b = GetDlgItemInt(*settingsWnd, IDC_EDIT_SGL_BLUE, NULL, false);
	if(r > 255)
		r = 255;
	if(g > 255)
		g = 255;
	if(b > 255)
		b = 255;
	byte red = r;
	byte green = g;
	byte blue = b;

	setItemText(IDC_EDIT_SGL_RED,	red);
	setItemText(IDC_EDIT_SGL_GREEN,	green);
	setItemText(IDC_EDIT_SGL_BLUE,	blue);

	if(USB != NULL && USB->valid())
		USB->setMode(SET_MODE_SINGLECOL, red, green, blue);
}

// Put into CPU usage mode and set brightness
void setUsageColour()
{
	unsigned int bright = GetDlgItemInt(*settingsWnd, IDC_EDIT_USAGE_BRT_MAX, NULL, false);
	if(bright > 255)
		bright = 255;
	byte brightness = bright;

	setItemText(IDC_EDIT_USAGE_BRT_MAX, brightness);
	SendMessage(*trackbar, TBM_SETPOS, true, brightness);

	if(USB != NULL && USB->valid())
		USB->setMode(SET_MODE_CPUUSAGE, brightness, 0, 0);
}

// Set transition time for CPU usage mode
void setTransitionTime()
{
	unsigned int time = GetDlgItemInt(*settingsWnd, IDC_EDIT_TRANS_TIME, NULL, false);
	if(time > 60000)
		time = 60000;

	// Due to the way the transition timing interval is worked out (byte value for interval time in milliseconds)
	// the total transition time is rounded down to the nearest multiple of 256,
	// so workout what the total transition time is acually going to be and show the user.
	time -= time % 256;

	setItemText(IDC_EDIT_TRANS_TIME, time);

	// Need to workout the time interval between each PWM duty cycle change and send that
	time = (unsigned int)floor((time / 256) + 0.5);

	if(USB != NULL && USB->valid())
		USB->setSetting(SET_TRANSITION_TIME, time);
}

DWORD WINAPI mainThread(LPVOID args)
{
//	int* value = reinterpret_cast<int*>(args);
//	printf("\n in thread %d", *value);

	CPU = new cpu();
	if(!CPU->cpuInit())
	{
		getchar();
		MessageBox(NULL, L"CPU Error", L"CPU Usage LEDs", MB_OK | MB_ICONERROR);
		exit(1);
	}

	char cpuData[7];
	int nBytes = 0;
	int usbData;

	USB = new usb(VEN_ID, VEN_NAME, DEV_ID, DEV_NAME);
	if(USB->valid() && settingsDlgOpen)
	{
		showCurrentSettings();
		showConnectionStatus();
	}
/*
	if(USB->handle == NULL)
	{
		fprintf(stderr, "Could not find USB device!\n");
		MessageBox(NULL, L"USB Error", L"CPU Usage LEDs", MB_OK | MB_ICONERROR);
		getchar();
		exit(1);
	}
*/
	while(1)
	{
		if(nBytes < 0 || !USB->valid())
		{
			//fprintf(stderr, "USB error: %s\n", usb_strerror());
//			cerr << "USB error: " << usb_strerror() << endl;
			showConnectionStatus();
			while(1)
			{
				delete USB;
				Sleep(500);
				USB = new usb(VEN_ID, VEN_NAME, DEV_ID, DEV_NAME);
//				if(!USB->valid())
//				{
//					//fprintf(stderr, "Could not find USB device!\n");
//					cerr << "Could not find USB device!" << endl;
//					continue;
//				}

				if(USB->valid())
				{
					if(settingsDlgOpen)
					{
						showCurrentSettings();
						showConnectionStatus();
					}
					break;
				}
			}
		}

		//Get initial Values
		if(!CPU->GetValues(true))
		{
//			cerr << "CPU usage error" << endl;
			Sleep(1000);
			continue;
		}

		Sleep(500);

		//Get current values
		if(!CPU->GetValues(false))
		{
//			cerr << "CPU usage2 error" << endl;
			continue;
		}
/*
		// Usage per core
		for(int i=0;i<CPU->cpuCount * 2;i+=2)
		{
			sprintf_s(cpuData,"%0.2f",CPU->calcUsage(i));
			cout << "PercentProcessorTime CPU " << (int)(i / 2) << ": \t\t" << cpuData << endl;
		}

		// Usage total
		sprintf_s(cpuData,"%0.2f",CPU->calcUsage(-1));	
		cout << "PercentProcessorTime CPU Total:\t\t" << cpuData << endl;
*/
		// Show CPU load %
		if(settingsDlgOpen)
			showCPULoad();

		// Send to USB
		// Expands 0.00 - 100.00 to 0 - 255
		// 14.26 -> 142.6 -> 142 -> map(val,0,1000,0,255) -> 28
		sprintf_s(cpuData,"%0.2f",CPU->calcUsage(-1));
		usbData = (int)(atof(cpuData) * 10);
//		cout << "Mapped: " << usbData << endl;
		usbData = map(usbData,0,1000,0,255);
		//usbData = atoi(cpuData);
		nBytes = USB->setCPUUsage(usbData);

//		cout << "USB: " << usbData << endl;
//		cout << "------" << endl;
	}

	return 1;
}

// This is the map function from Arduino
byte map(int x, int in_min, int in_max, int out_min, int out_max)
{
	return (byte)(((long)x - in_min) * (out_max - out_min) / (in_max - in_min) + out_min);
}

// Update boxes with current device settings
void showCurrentSettings()
{
	if(USB != NULL && USB->valid())
	{
		USB->getDeviceState();
		
		setItemText(IDC_EDIT_SGL_RED,	USB->deviceRGBVal.red);
		setItemText(IDC_EDIT_SGL_GREEN,	USB->deviceRGBVal.green);
		setItemText(IDC_EDIT_SGL_BLUE,	USB->deviceRGBVal.blue);

		setItemText(IDC_EDIT_USAGE_BRT_MAX, USB->deviceSettings.maxBrightness);
		SendMessage(*trackbar, TBM_SETPOS, true, USB->deviceSettings.maxBrightness);

		unsigned int transTime = USB->deviceSettings.transitionTime * 256;
		setItemText(IDC_EDIT_TRANS_TIME, transTime);
	}
}

// Show connection status and firmware version
// showCurrentSettings() must be called before this function
void showConnectionStatus()
{
	if(USB != NULL && USB->valid())
	{
		SetDlgItemText(*settingsWnd, IDC_STATIC_USB_STATUS, L"Connected");
		char buff[17];
		sprintf_s(buff,"Firmware: %02u.%02u", USB->version[0], USB->version[1]);
		wchar_t widearray[17];
		mbstowcs_s(NULL, widearray, buff, 16);
		SetDlgItemText(*settingsWnd, IDC_STATIC_USB_FIRMWARE, widearray);
	}
	else
	{
		SetDlgItemText(*settingsWnd, IDC_STATIC_USB_STATUS, L"Disconnected");
		SetDlgItemText(*settingsWnd, IDC_STATIC_USB_FIRMWARE, L"Firmware: 00.00");
	}
}

// Show CPU load
void showCPULoad()
{
	char buff[12];
	sprintf_s(buff,"CPU: %0.2f", CPU->calcUsage(-1));
	wchar_t widearray[12];
	mbstowcs_s(NULL, widearray, buff, 11);
	SetDlgItemText(*settingsWnd, IDC_STATIC_CPU_USAGE, widearray);
}

void setItemText(int item, int text)
{
	char newtext[11];
	_itoa_s(text,newtext,10);
	wchar_t widearray[11];
	mbstowcs_s(NULL, widearray, newtext, 11);
	SetDlgItemText(*settingsWnd, item, widearray);
}

