/*
 * Project: AVR ATtiny CPU Usage LEDs
 * Author: Zak Kemble, me@zakkemble.co.uk
 * Copyright: (C) 2012 by Zak Kemble
 * License: GNU GPL v3 (see License.txt)
 */

#include "usb.h"

usb::usb(int _vendor, char* _vendorName, int _product, char* _productName)
{
	handle		= NULL;
	vendor		= _vendor;
	vendorName	= _vendorName;
	product		= _product;
	productName	= _productName;
	usbOpenDevice();
//	if(handle != NULL)
//		getDeviceState();
}

usb::~usb()
{
	if(handle != NULL)
		usb_close(handle);
}

bool usb::valid()
{
	return (handle != NULL);
}

int usb::setCPUUsage(int data)
{
	return send(USB_REQ_CPUUSAGE, data);
}

void usb::setMode(byte mode, byte _data1, byte _data2, byte _data3)
{
	int data1 = (_data1<<8) | mode;		// Mode and red value
	int data2 = (_data3<<8) | _data2;	// Green and blue values
	send(USB_REQ_SETTING, data1, data2);
}

void usb::setSetting(byte setting, byte _data1)
{
	int data1 = (_data1<<8) | setting; // What setting to set and its new value
	send(USB_REQ_SETTING, data1);
}

// Get firmware version and settings
void usb::getDeviceState()
{
	char buffer[8];
	if(send(USB_REQ_DEVSTATE, 0, 0, buffer, sizeof(buffer)))
	{
		version[0]						= buffer[1];
		version[1]						= buffer[0];
		deviceSettings.ledMode			= buffer[2];
		deviceSettings.maxBrightness	= buffer[3];
		deviceSettings.transitionTime	= buffer[4];
		deviceRGBVal.red				= buffer[5];
		deviceRGBVal.green				= buffer[6];
		deviceRGBVal.blue				= buffer[7];
	}
}

int usb::send(byte reqType, int data1)
{
	return send(reqType, data1, 0, NULL, 0);
}

int usb::send(byte reqType, int data1, int data2)
{
	return send(reqType, data1, data2, NULL, 0);
}

int usb::send(byte reqType, int data1, int data2, char* out, int size)
{
	return usb_control_msg(handle,
		USB_TYPE_VENDOR | USB_RECIP_DEVICE | USB_ENDPOINT_IN,
		reqType, data1, data2, out, size, 2000);
}


/**
 * Project: AVR ATtiny USB Tutorial at http://codeandlife.com/
 * Author: Joonas Pihlajamaa, joonas.pihlajamaa@iki.fi
 * Based on V-USB example code by Christian Starkjohann
 * Copyright: (C) 2012 by Joonas Pihlajamaa
 * License: GNU GPL v3 (see License.txt)
 */

void usb::usbOpenDevice()
{
	struct usb_bus *bus;
	struct usb_device *dev;
	char devVendor[256], devProduct[256];
    
	usb_dev_handle* _handle = NULL;
	
	usb_init();
	usb_find_busses();
	usb_find_devices();
	
	for(bus=usb_get_busses(); bus; bus=bus->next)
	{
		for(dev=bus->devices; dev; dev=dev->next)
		{
			if(dev->descriptor.idVendor != vendor || dev->descriptor.idProduct != product)
                continue;

            // we need to open the device in order to query strings 
            if(!(_handle = usb_open(dev)))
			{
                fprintf(stderr, "Warning: cannot open USB device: %s\n", usb_strerror());
                continue;
            }
            
            // get vendor name 
            if(usbGetDescriptorString(_handle, dev->descriptor.iManufacturer, 0x0409, devVendor, sizeof(devVendor)) < 0)
			{
                fprintf(stderr, "Warning: cannot query manufacturer for device: %s\n", usb_strerror());
				usb_close(_handle);
                continue;
            }
            
            // get product name 
            if(usbGetDescriptorString(_handle, dev->descriptor.iProduct, 0x0409, devProduct, sizeof(devVendor)) < 0)
			{
                fprintf(stderr, "Warning: cannot query product for device: %s\n", usb_strerror());
                usb_close(_handle);
                continue;
            }
            
            if(strcmp(devVendor, vendorName) == 0 && strcmp(devProduct, productName) == 0)
                handle = _handle;
            else
                usb_close(_handle);
		}
	}
}

// used to get descriptor strings for device identification 
int usb::usbGetDescriptorString(usb_dev_handle *dev, int index, int langid, char *buf, int buflen)
{
    char buffer[256];
    int rval, i;

	// make standard request GET_DESCRIPTOR, type string and given index 
    // (e.g. dev->iProduct)
	rval = usb_control_msg(dev, 
        USB_TYPE_STANDARD | USB_RECIP_DEVICE | USB_ENDPOINT_IN, 
        USB_REQ_GET_DESCRIPTOR, (USB_DT_STRING << 8) + index, langid, 
        buffer, sizeof(buffer), 1000);
        
    if(rval < 0) // error
		return rval;
	
    // rval should be bytes read, but buffer[0] contains the actual response size
	if((unsigned char)buffer[0] < rval)
		rval = (unsigned char)buffer[0]; // string is shorter than bytes read
	
	if(buffer[1] != USB_DT_STRING) // second byte is the data type
		return 0; // invalid return type
		
	// we're dealing with UTF-16LE here so actual chars is half of rval,
	// and index 0 doesn't count
	rval /= 2;
	
	// lossy conversion to ISO Latin1 
	for(i = 1; i < rval && i < buflen; i++)
	{
		if(buffer[2 * i + 1] == 0)
			buf[i-1] = buffer[2 * i];
		else
			buf[i-1] = '?'; // outside of ISO Latin1 range
	}
	buf[i-1] = 0;
	
	return i-1;
}
