/*
 * Project: AVR ATtiny CPU Usage LEDs
 * Author: Zak Kemble, me@zakkemble.co.uk
 * Copyright: (C) 2012 by Zak Kemble
 * License: GNU GPL v3 (see License.txt)
 */

#include "cpuUsageLed.h"

int main(void)
{
#if CFG_WDT_DEBUG
	byte reset = (MCUSR & RESET_MASK);
#endif

	setupHardware();
	loadUSBOscCalibration();
//	usbDeviceDisconnect();

#if CFG_WDT_DEBUG
	// Error reset happened
	if(reset)
		resetError();
#endif

	delay(250);

//	usbDeviceConnect();
	usbInit();
	sei();
#if CFG_WDT_ENABLE
	wdt_reset();
#endif

	flashTest();

	// Load settings from EEPROM
	eeprom_read_block(&settings, &eepSettings, sizeof(settingsS));
	eeprom_read_block(&currentColour, &eepRgbVal, sizeof(rgbVal));

	while(1)
	{
		// timerTick is set to true approx every millisecond
		if(timerTick)
		{
			// Colour update timer
			++brightness.counter;
			if(brightness.counter > settings.transitionTime)
			{
				if(settings.ledMode == MODE_CPU_USAGE)
					linearTrans();
				setBrightness(currentColour.red,	&LEDRedSet);
				setBrightness(currentColour.green,	&LEDGreenSet);
				setBrightness(currentColour.blue,	&LEDBlueSet);
				brightness.counter = 0;
			}

			// EEPROM save delay timer
			if(eepSave.enabled)
			{
				++eepSave.counter;
				if(eepSave.counter > SAVE_DELAY_TIME)
				{
					eeprom_update_block(&settings, (settingsS*)&eepSettings, sizeof(settingsS));
					eeprom_update_block(&currentColour, (rgbVal*)&eepRgbVal, sizeof(rgbVal));
					eepSave.enabled = false;
					eepSave.counter = 0;
				}
			}

			// Idle timer
			if(settings.ledMode == MODE_CPU_USAGE)
			{
				++idle.counter;
				if(!idle.enabled && idle.counter > IDLE_TIME)
				{
					idle.enabled = true;
					cpuUsage.target = MAX_RGB_LEVEL;
				}
				else if(idle.enabled && cpuUsage.current == cpuUsage.target)
					cpuUsage.target = ~cpuUsage.target;
			}

			timerTick = false;
		}

		usbPoll();

#if CFG_WDT_ENABLE
		if(resetWDT)
		{
			wdt_int_reset();
			resetWDT = false;
		}
#endif

		// Sleep mode makes USB go wonky
//		sleep_mode();
	}
}

static void setupHardware()
{
#if CFG_WDT_ENABLE
	// Set watchdog
	wdt_enable(WDTO_2S);
	MCUSR = 0x00;
	wdt_int_reset();
#endif

	// Set frequency and power
	clock_prescale_set(CPU_DIV);
	power_usi_disable();
	power_adc_disable();
//	set_sleep_mode(SLEEP_MODE_IDLE);

	// Configure IO pins
	LED_RED_DDR |= (1<<LED_RED_PIN);
	LED_GREEN_DDR |= (1<<LED_GREEN_PIN);
	LED_BLUE_DDR |= (1<<LED_BLUE_PIN);

	// Timer settings
	TCCR0A = (1<<WGM00)|(1<<WGM01)|(1<<COM0A1)|(1<<COM0B1);
	TCCR0B = (1<<CS00)|(1<<CS01);
	TIMSK = (1<<TOIE0);
	TCCR1 = (1<<CS12)|(1<<CS11)|(1<<CS10);
	GTCCR = (1<<PWM1B)|(1<<COM1B1);
}

static void setBrightness(byte level, void (*ledFuncSet)(byte))
{
	// Don't modify RGB levels for single colour mode
	if(settings.ledMode == MODE_CPU_USAGE)
	{
		// Gamma correction would go around here somewhere

		// Get sine level
		level = getSineVal(level);

		// Adjust for brightness
		level = map(level,MIN_RGB_LEVEL,MAX_RGB_LEVEL,MIN_RGB_LEVEL,settings.maxBrightness);
	}

	// Set brightness
	(*ledFuncSet)(level);
}

// Makes a smoother transition between colours
static byte getSineVal(byte idx)
{
	byte origIdx = idx;
	if(idx > 127)
		idx -= 128;
	else
		idx = (idx - 127) * -1; // Use ABS? Increases program size a bit
	byte val = pgm_read_byte(&sineVals[idx]);
	
	if(origIdx > 127)
		val += 127;
	else
		val = (val - 127) * -1; // Use ABS? Increases program size a bit
	return val;
}

static void linearTrans()
{
	if(cpuUsage.target > cpuUsage.current && cpuUsage.current < 255)
		++cpuUsage.current;
	else if(cpuUsage.target < cpuUsage.current && cpuUsage.current > 0)
		--cpuUsage.current;

	updateColours();
}

static void updateColours()
{
	currentColour.red = getColour(128,255,MIN_RGB_LEVEL,MAX_RGB_LEVEL);
	currentColour.blue = getColour(0,127,MAX_RGB_LEVEL,MIN_RGB_LEVEL);

	if(cpuUsage.current < 128)
		currentColour.green = getColour(0,127,MIN_RGB_LEVEL,MAX_RGB_LEVEL);
	else
		currentColour.green = getColour(128,255,MAX_RGB_LEVEL,MIN_RGB_LEVEL);
}

static byte getColour(byte inMin, byte inMax, byte outMin, byte outMax)
{
	return map(constrain(cpuUsage.current,inMin,inMax),inMin,inMax,outMin,outMax);
}

// This is the map function from Arduino, modified for byte values instead of long
static byte map(byte x, byte in_min, byte in_max, byte out_min, byte out_max)
{
// At least one of the values must be 32 bit, in this case x is type casted to long since some parts of the calculation can return values down to -65025.
// Or something like that... without the type cast the return value isn't quite right, map(255,0,255,0,255) would return 254

	return (((long)x - in_min) * (out_max - out_min) / (in_max - in_min) + out_min);
}

// This is the constrain function from Arduino, modified for byte values instead of long
static byte constrain(byte val, byte min, byte max)
{
	if(val < min)
		return min;
	else if(val > max)
		return max;
	else
		return val;
}

// Flash each colour when first turning on 
static void flashTest()
{
	LEDGreenSet(0);
	LEDBlueSet(0);
	LEDRedSet(255); // Red
	delay_usbpoll(100);
	LEDRedSet(0);
	LEDGreenSet(255); // Green
	delay_usbpoll(100);
	LEDGreenSet(0);
	LEDBlueSet(255); // Blue
	delay_usbpoll(100);
	for(byte i=0;i<5;++i) // White flash
	{
		LEDRedSet(255);
		LEDGreenSet(255);
		LEDBlueSet(255);
		delay_usbpoll(40);
		LEDRedSet(0);
		LEDGreenSet(0);
		LEDBlueSet(0);
		delay_usbpoll(40);
	}
	delay_usbpoll(100);
}

// Delay while still polling USB
static void delay_usbpoll(byte ms)
{
	while(ms--)
	{
		while(!timerTick)
			usbPoll();
		timerTick = false;
	}

#if CFG_WDT_ENABLE
	wdt_reset();
#endif
}

static void LEDRedSet(byte brightness)
{
// Timer1 is only used for red, turn it off when not needed.

	if(brightness == 0 || brightness == 255)
	{
		LED_RED_PORT = (LED_RED_PORT & ~(1<<LED_RED_PIN)) | ((!!brightness)<<LED_RED_PIN);
		GTCCR &= ~(1<<COM1B1);
		power_timer1_disable();
	}
	else
	{
		power_timer1_enable();
		GTCCR |= (1<<COM1B1);
		LED_RED_PWM = brightness;
	}
}

static void LEDGreenSet(byte brightness)
{
	if(brightness == 0 || brightness == 255)
	{
		LED_GREEN_PORT = (LED_GREEN_PORT & ~(1<<LED_GREEN_PIN)) | ((!!brightness)<<LED_GREEN_PIN);
		TCCR0A &= ~(1<<COM0B1);
	}
	else
	{
		TCCR0A |= (1<<COM0B1);
		LED_GREEN_PWM = brightness;
	}
}

static void LEDBlueSet(byte brightness)
{
	if(brightness == 0 || brightness == 255)
	{
		LED_BLUE_PORT = (LED_BLUE_PORT & ~(1<<LED_BLUE_PIN)) | ((!!brightness)<<LED_BLUE_PIN);
		TCCR0A &= ~(1<<COM0A1);
	}
	else
	{
		TCCR0A |= (1<<COM0A1);
		LED_BLUE_PWM = brightness;
	}
}

#if CFG_WDT_DEBUG
// Flash red if the controller was restarted by the watchdog
static void resetError()
{
	LEDGreenSet(0);
	LEDBlueSet(0);
	while(1)
	{
		wdt_reset();
		delay(100);
		LEDRedSet(255);
		delay(100);
		LEDRedSet(0);
	}
}
#endif

// 16MHz / 64 prescale = 250KHz timer clock
// 250000 / 256 for overflow = ~976 overflows per second
// ~976 state changes per sec = ~500Hz
// Each overflow is just a tiny bit more than a millisecond
// Actually we're clocked at 16.5MHz causing each overflow to be a tiny bit less than a millisecond
ISR(TIMER0_OVF_vect)
{
	timerTick = true;
}

ISR(WDT_vect)
{
#if CFG_WDT_ENABLE
	resetWDT = true;
#endif
}
