/*
 * Project: AVR ATtiny CPU Usage LEDs
 * Author: Zak Kemble, me@zakkemble.co.uk
 * Copyright: (C) 2012 by Zak Kemble
 * License: GNU GPL v3 (see License.txt)
 */

#include "mainBackend.h"

HWND* settingsWnd;
HWND* trackbar;
bool* settingsDlgOpen;
usb* USB = NULL;

void backend(HWND* _settingsWnd, HWND* _trackbar, bool* _settingsDlgOpen)
{
	settingsWnd		= _settingsWnd;
	trackbar		= _trackbar;
	settingsDlgOpen	= _settingsDlgOpen;
	DWORD threadId;
	HANDLE hThread	= CreateThread(NULL, 0, mainThread, NULL, 0, &threadId);
}

// Put into single colour mode and set the colour
void setSingleColour()
{
	unsigned int r = GetDlgItemInt(*settingsWnd, IDC_EDIT_SGL_RED, NULL, false);
	unsigned int g = GetDlgItemInt(*settingsWnd, IDC_EDIT_SGL_GREEN, NULL, false);
	unsigned int b = GetDlgItemInt(*settingsWnd, IDC_EDIT_SGL_BLUE, NULL, false);
	if(r > 255)
		r = 255;
	if(g > 255)
		g = 255;
	if(b > 255)
		b = 255;
	byte red = r;
	byte green = g;
	byte blue = b;

	setItemText(IDC_EDIT_SGL_RED,	red);
	setItemText(IDC_EDIT_SGL_GREEN,	green);
	setItemText(IDC_EDIT_SGL_BLUE,	blue);

	if(deviceValid(USB))
		USB->setMode(SET_MODE_SINGLECOL, red, green, blue);
}

// Put into CPU usage mode and set brightness
void setUsageColour()
{
	unsigned int bright = GetDlgItemInt(*settingsWnd, IDC_EDIT_USAGE_BRT_MAX, NULL, false);
	if(bright > 255)
		bright = 255;
	byte brightness = bright;

	setItemText(IDC_EDIT_USAGE_BRT_MAX, brightness);
	SendMessage(*trackbar, TBM_SETPOS, true, brightness);

	if(deviceValid(USB))
		USB->setMode(SET_MODE_CPUUSAGE, brightness, 0, 0);
}

// Set transition time for CPU usage mode
void setTransitionTime()
{
	unsigned int time = GetDlgItemInt(*settingsWnd, IDC_EDIT_TRANS_TIME, NULL, false);
	if(time > 60000)
		time = 60000;

	// Due to the way the transition timing interval is worked out (byte value for interval time in milliseconds)
	// the total transition time is rounded down to the nearest multiple of 256,
	// so workout what the total transition time is acually going to be and show the user.
	time -= time % 256;

	setItemText(IDC_EDIT_TRANS_TIME, time);

	// Need to workout the time interval between each PWM duty cycle change and send that
	time = (unsigned int)floor((time / 256) + 0.5);

	if(deviceValid(USB))
		USB->setSetting(SET_TRANSITION_TIME, time);
}

DWORD WINAPI mainThread(LPVOID args)
{
	int nBytes = 0;
	byte cpuUsage;

	while(1)
	{
		Sleep(500);

		// Get usage
		cpuUsage = getCPUUsage();

		// Show CPU usage
		if(*settingsDlgOpen)
			showCPULoad(cpuUsage);

		// See of the device is ok and connected, if not, try reconnecting
		if(nBytes < 0 || !deviceValid(USB))
		{
			delete USB;
			USB = NULL;
			USB = new usb(VEN_ID, VEN_NAME, DEV_ID, DEV_NAME);
			if(!deviceValid(USB))
			{
				if(*settingsDlgOpen)
					showConnectionStatus();
				continue;
			}
			else if(*settingsDlgOpen)
			{
				showCurrentSettings();
				showConnectionStatus();
			}
		}

		// Send to USB
		nBytes = USB->setCPUUsage(cpuUsage);
	}

	return 1;
}

// Get CPU usage and return as a value from 0 to 255
byte getCPUUsage()
{
	static __int64 last_idleTime	= 0;
	static __int64 last_kernelTime	= 0;
	static __int64 last_userTime	= 0;

	__int64 idleTime;
	__int64 kernelTime;
	__int64 userTime;
	BOOL res = GetSystemTimes((LPFILETIME)&idleTime, (LPFILETIME)&kernelTime, (LPFILETIME)&userTime);

	__int64 usr = userTime - last_userTime;
	__int64 ker = kernelTime - last_kernelTime;
	__int64 idl = idleTime - last_idleTime;
	__int64 sys = ker + usr;

	last_idleTime = idleTime;
	last_kernelTime = kernelTime;
	last_userTime = userTime;

	if(sys == 0)
		return 0;

	return (byte)((sys - idl) * 255 / sys);
}

bool deviceValid(usb* device)
{
	return (device != NULL && device->valid());
}

// Update boxes with current device settings
void showCurrentSettings()
{
	if(deviceValid(USB))
	{
		USB->getDeviceState();
		
		setItemText(IDC_EDIT_SGL_RED,	USB->deviceRGBVal.red);
		setItemText(IDC_EDIT_SGL_GREEN,	USB->deviceRGBVal.green);
		setItemText(IDC_EDIT_SGL_BLUE,	USB->deviceRGBVal.blue);

		setItemText(IDC_EDIT_USAGE_BRT_MAX, USB->deviceSettings.maxBrightness);
		SendMessage(*trackbar, TBM_SETPOS, true, USB->deviceSettings.maxBrightness);

		unsigned int transTime = USB->deviceSettings.transitionTime * 256;
		setItemText(IDC_EDIT_TRANS_TIME, transTime);
	}
}

// Show connection status and firmware version
// showCurrentSettings() must be called before this function
void showConnectionStatus()
{
	if(deviceValid(USB))
	{
		SetDlgItemText(*settingsWnd, IDC_STATIC_USB_STATUS, L"Connected");
		char buff[17];
		sprintf_s(buff,"Firmware: %02u.%02u", USB->version[0], USB->version[1]);
		wchar_t widearray[17];
		mbstowcs_s(NULL, widearray, buff, 16);
		SetDlgItemText(*settingsWnd, IDC_STATIC_USB_FIRMWARE, widearray);
	}
	else
	{
		SetDlgItemText(*settingsWnd, IDC_STATIC_USB_STATUS, L"Disconnected");
		SetDlgItemText(*settingsWnd, IDC_STATIC_USB_FIRMWARE, L"Firmware: 00.00");
	}
}

// Show CPU load
void showCPULoad(byte cpuUsage)
{
	char buff[10];
	sprintf_s(buff,"CPU: %0u", map(cpuUsage,0,255,0,100));
	wchar_t widearray[10];
	mbstowcs_s(NULL, widearray, buff, 9);
	SetDlgItemText(*settingsWnd, IDC_STATIC_CPU_USAGE, widearray);
}

// This is the map function from Arduino
byte map(int x, int in_min, int in_max, int out_min, int out_max)
{
	return (byte)(((long)x - in_min) * (out_max - out_min) / (in_max - in_min) + out_min);
}

void setItemText(int item, int text)
{
	char newtext[11];
	_itoa_s(text,newtext,10);
	wchar_t widearray[11];
	mbstowcs_s(NULL, widearray, newtext, 11);
	SetDlgItemText(*settingsWnd, item, widearray);
}

