/*
 * Project: AVR ATtiny CPU Usage LEDs
 * Author: Zak Kemble, me@zakkemble.co.uk
 * Copyright: (C) 2012 by Zak Kemble
 * License: GNU GPL v3 (see License.txt)
 */

#include "config.h"
#include "util.h"
#include "typedefs.h"
#include "cpuUsageLed.h"
#include "sine.h"
#include "usb.h"
#include "usbdrv/usbdrv.h"
#include "usbdrv/oddebug.h"
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/power.h>
#include <avr/wdt.h>
#include <avr/eeprom.h>
//#include <avr/sleep.h>
#include <util/delay.h>

static volatile bool timerTick;
s_rgbVal currentColour;
s_cpuUsage cpuUsage;
s_settings settings;
s_timer idle					= {false,0};
s_timer eepSave					= {false,0};
static s_timer brightness		= {true,0};
#if CFG_WDT_ENABLE
static volatile bool resetWDT	= false;
#endif
static const s_settings eepSettings EEMEM		= {255,MODE_CPU_USAGE,6,5};
static const s_rgbVal eepRgbVal EEMEM			= {0,0,255};

static void init();
static void eppLoadSettings();
static void checkTimerTick();
static void brightnessTimer();
static void eepSaveTimer();
static void idleTimer();
static void setBrightness(byte, void (*ledFuncSet)(byte));
static byte getSineVal(byte);
static void linearTrans();
static void updateColours();
static byte getColour(byte, byte, byte,byte);
static void flashTest();
static void delay_usbpoll(byte);
static void LEDRedSet(byte);
static void LEDGreenSet(byte);
static void LEDBlueSet(byte);
#if CFG_WDT_DEBUG
static void resetError();
#endif

int main(void)
{
#if CFG_WDT_DEBUG
	byte WDTreset = (MCUSR & (1<<WDRF));
#endif

	init();
	loadUSBOscCalibration();
//	usbDeviceDisconnect();

#if CFG_WDT_DEBUG
	// Error reset happened
	if(WDTreset)
		resetError();
#endif

	delay(250);

//	usbDeviceConnect();
	usbInit();
	sei();

#if CFG_WDT_ENABLE
	wdt_reset();
#endif

	flashTest();
	eppLoadSettings();

	while(1)
	{
		checkTimerTick();
		usbPoll();

#if CFG_WDT_ENABLE
		if(resetWDT)
		{
			wdt_int_reset();
			resetWDT = false;
		}
#endif

		// Sleep mode makes USB go wonky
//		sleep_mode();
	}
}

static void init()
{
#if CFG_WDT_ENABLE
	// Set watchdog
	wdt_enable(WDTO_2S);
	MCUSR &= ~(1<<WDRF);
	wdt_int_reset();
#endif

	// Set frequency and power
	clock_prescale_set(CPU_DIV);
	power_usi_disable();
	power_adc_disable();
//	set_sleep_mode(SLEEP_MODE_IDLE);

	// Configure IO pins
	LED_RED_DDR |= (1<<LED_RED_BIT);
	LED_GREEN_DDR |= (1<<LED_GREEN_BIT);
	LED_BLUE_DDR |= (1<<LED_BLUE_BIT);

	// Timer settings
	TCCR0A = (1<<WGM00)|(1<<WGM01)|(1<<COM0A1)|(1<<COM0B1);
	TCCR0B = (1<<CS00)|(1<<CS01);
	TIMSK = (1<<TOIE0);
	TCCR1 = (1<<CS12)|(1<<CS11)|(1<<CS10);
	GTCCR = (1<<PWM1B)|(1<<COM1B1);
}

static void eppLoadSettings()
{
	// Load settings from EEPROM
	eeprom_read_block(&settings, &eepSettings, sizeof(s_settings));
	eeprom_read_block(&currentColour, &eepRgbVal, sizeof(s_rgbVal));
}

static void checkTimerTick()
{
	// timerTick is set to true approx every millisecond
	if(timerTick)
	{
		brightnessTimer();
		eepSaveTimer();
		idleTimer();
		timerTick = false;
	}
}

static void brightnessTimer()
{
	// Colour update timer
	brightness.counter++;
	if(brightness.counter > settings.transitionTime)
	{
		if(settings.ledMode == MODE_CPU_USAGE)
			linearTrans();
		setBrightness(currentColour.red,	&LEDRedSet);
		setBrightness(currentColour.green,	&LEDGreenSet);
		setBrightness(currentColour.blue,	&LEDBlueSet);
		brightness.counter = 0;
	}
}

static void eepSaveTimer()
{
	// EEPROM save delay timer
	if(eepSave.enabled)
	{
		eepSave.counter++;
		if(eepSave.counter > SAVE_DELAY_TIME)
		{
			eeprom_update_block(&settings, (s_settings*)&eepSettings, sizeof(s_settings));
			eeprom_update_block(&currentColour, (s_rgbVal*)&eepRgbVal, sizeof(s_rgbVal));
			eepSave.enabled = false;
			eepSave.counter = 0;
		}
	}
}

static void idleTimer()
{
	// Idle timer
	if(settings.ledMode == MODE_CPU_USAGE && settings.idleTime != 0)
	{
		idle.counter++;
		if(!idle.enabled && idle.counter > settings.idleTime * 1000)
		{
			idle.enabled = true;
			cpuUsage.target = MAX_RGB_LEVEL;
		}
		else if(idle.enabled && cpuUsage.current == cpuUsage.target)
			cpuUsage.target = ~cpuUsage.target;
	}
}

static void setBrightness(byte level, void (*ledFuncSet)(byte))
{
	// Only modify RGB levels for CPU usage mode
	if(settings.ledMode == MODE_CPU_USAGE)
	{
		// Gamma correction would go around here somewhere

		// Adjust to sine level
		level = getSineVal(level);

		// Adjust for brightness
		level = map(level, MIN_RGB_LEVEL, MAX_RGB_LEVEL, MIN_RGB_LEVEL, settings.maxBrightness);
	}

	// Set brightness
	(*ledFuncSet)(level);
}

// Makes a smoother transition between colours
static byte getSineVal(byte idx)
{
	byte origIdx = idx;
	if(idx > 127)
		idx -= 128;
	else
		idx = (idx - 127) * -1; // Use abs()? Increases program size a bit
	byte val = pgm_read_byte(&sineVals[idx]);
	
	if(origIdx > 127)
		val += 127;
	else
		val = (val - 127) * -1; // Use abs()? Increases program size a bit
	return val;
}

// Transition from one CPU level to another
static void linearTrans()
{
	if(cpuUsage.target > cpuUsage.current && cpuUsage.current < 255)
		cpuUsage.current++;
	else if(cpuUsage.target < cpuUsage.current && cpuUsage.current > 0)
		cpuUsage.current--;

	updateColours();
}

static void updateColours()
{
	currentColour.red	= getColour(128, 255, MIN_RGB_LEVEL, MAX_RGB_LEVEL);
	currentColour.blue	= getColour(0, 127, MAX_RGB_LEVEL, MIN_RGB_LEVEL);

	if(cpuUsage.current < 128)
		currentColour.green = getColour(0, 127, MIN_RGB_LEVEL, MAX_RGB_LEVEL);
	else
		currentColour.green = getColour(128, 255, MAX_RGB_LEVEL, MIN_RGB_LEVEL);
}

static byte getColour(byte inMin, byte inMax, byte outMin, byte outMax)
{
	return map(constrain(cpuUsage.current, inMin, inMax), inMin, inMax, outMin, outMax);
}

// Flash each colour when first turning on 
static void flashTest()
{
	LEDGreenSet(0);
	LEDBlueSet(0);
	LEDRedSet(255); // Red
	delay_usbpoll(100);
	LEDRedSet(0);
	LEDGreenSet(255); // Green
	delay_usbpoll(100);
	LEDGreenSet(0);
	LEDBlueSet(255); // Blue
	delay_usbpoll(100);
	for(byte i=0;i<5;++i) // White flash
	{
		LEDRedSet(255);
		LEDGreenSet(255);
		LEDBlueSet(255);
		delay_usbpoll(40);
		LEDRedSet(0);
		LEDGreenSet(0);
		LEDBlueSet(0);
		delay_usbpoll(40);
	}
	delay_usbpoll(100);
}

// Delay while still polling USB
static void delay_usbpoll(byte ms)
{
	while(ms--)
	{
		while(!timerTick)
			usbPoll();
		timerTick = false;
	}

#if CFG_WDT_ENABLE
	wdt_reset();
#endif
}

static void LEDRedSet(byte brightness)
{
// Timer1 is only used for red, turn it off when not needed.

	if(brightness == 0 || brightness == 255)
	{
		pinWrite(&LED_RED_PORT, LED_RED_BIT, brightness);
		GTCCR &= ~(1<<COM1B1);
		power_timer1_disable();
	}
	else
	{
		power_timer1_enable();
		GTCCR |= (1<<COM1B1);
		LED_RED_PWM = brightness;
	}
}

static void LEDGreenSet(byte brightness)
{
	if(brightness == 0 || brightness == 255)
	{
		pinWrite(&LED_GREEN_PORT, LED_GREEN_BIT, brightness);
		TCCR0A &= ~(1<<COM0B1);
	}
	else
	{
		TCCR0A |= (1<<COM0B1);
		LED_GREEN_PWM = brightness;
	}
}

static void LEDBlueSet(byte brightness)
{
	if(brightness == 0 || brightness == 255)
	{
		pinWrite(&LED_BLUE_PORT, LED_BLUE_BIT, brightness);
		TCCR0A &= ~(1<<COM0A1);
	}
	else
	{
		TCCR0A |= (1<<COM0A1);
		LED_BLUE_PWM = brightness;
	}
}

#if CFG_WDT_DEBUG
// Flash red if the controller was restarted by the watchdog
static void resetError()
{
	LEDGreenSet(0);
	LEDBlueSet(0);
	while(1)
	{
		wdt_reset();
		delay(100);
		LEDRedSet(255);
		delay(100);
		LEDRedSet(0);
	}
}
#endif

// 16MHz / 64 prescale = 250KHz timer clock
// 250000 / 256 for overflow = ~976 overflows per second
// ~976 state changes per sec = ~500Hz
// Each overflow is just a tiny bit more than a millisecond
// Actually we're clocked at 16.5MHz so each overflow to be a tiny bit less than a millisecond
ISR(TIMER0_OVF_vect)
{
	timerTick = true;
}

ISR(WDT_vect)
{
#if CFG_WDT_ENABLE
	resetWDT = true;
#endif
}
