/*
 * Project: AVR ATtiny CPU Usage LEDs
 * Author: Zak Kemble, me@zakkemble.co.uk
 * Copyright: (C) 2012 by Zak Kemble
 * License: GNU GPL v3 (see License.txt)
 */

#include "config.h"
#include "typedefs.h"
#include "usb.h"
#include "cpuUsageLed.h"
#include "usbdrv/usbdrv.h"
#include <avr/eeprom.h>
#include <avr/interrupt.h>

extern s_rgbVal currentColour;
extern s_cpuUsage cpuUsage;
extern s_settings settings;
extern s_timer idle;
extern s_timer eepSave;

static const byte EEMEM eepCalibration = 0xFF;

static void getDeviceState();
static void setCPUUsage(byte);
static void setSetting(byte[], byte[]);
static void calibrateOscillator();

static void getDeviceState()
{
	static byte stateData[9] = {USB_CFG_DEVICE_VERSION};
	stateData[2] = settings.ledMode;
	stateData[3] = settings.maxBrightness;
	stateData[4] = settings.transitionTime;
	stateData[5] = settings.idleTime;
	stateData[6] = currentColour.red;
	stateData[7] = currentColour.green;
	stateData[8] = currentColour.blue;
	usbMsgPtr = stateData;
}

static void setCPUUsage(byte usage)
{
	cpuUsage.target = usage;
}

static void setSetting(byte wValue[], byte wIndex[])
{
	bool eep = true;
	switch(wValue[0])
	{
		case SET_MODE_CPUUSAGE:
			settings.ledMode = MODE_CPU_USAGE;
			settings.maxBrightness = wValue[1];
			break;
		case SET_MODE_SINGLECOL:
			settings.ledMode = MODE_SINGLE_COL;
			currentColour.red = wValue[1];
			currentColour.green = wIndex[0];
			currentColour.blue = wIndex[1];
			break;
		case SET_TRANSITION_TIME:
			settings.transitionTime = wValue[1];
			break;
		case SET_IDLE_TIME:
			settings.idleTime = wValue[1];
			break;
		default:
			eep = false;
			break;
	}

	if(eep)
	{
		eepSave.counter = 0;
		eepSave.enabled = true;
	}
}

USB_PUBLIC uchar usbFunctionSetup(uchar data[8])
{
	idle.enabled = false;
	idle.counter = 0;

	usbRequest_t *rq = (void*)data;
	switch(rq->bRequest)
	{
		case USB_REQ_DEVSTATE:
			getDeviceState();
			return 9;
		case USB_REQ_CPUUSAGE:
			setCPUUsage(rq->wValue.bytes[0]);
			break;
		case USB_REQ_SETTING:
			setSetting(rq->wValue.bytes, rq->wIndex.bytes);
			break;
		default:
			break;
	}

    return 0;
}




/* Name: osccal.c
 * Author: Christian Starkjohann
 * Creation Date: 2008-04-10
 * Tabsize: 4
 * Copyright: (c) 2008 by OBJECTIVE DEVELOPMENT Software GmbH
 * License: GNU GPL v2 (see License.txt), GNU GPL v3 or proprietary (CommercialLicense.txt)
 * This Revision: $Id$
 */

/* ------------------------------------------------------------------------- */
/* ------------------------ Oscillator Calibration ------------------------- */
/* ------------------------------------------------------------------------- */

/* Calibrate the RC oscillator to 8.25 MHz. The core clock of 16.5 MHz is
 * derived from the 66 MHz peripheral clock by dividing. Our timing reference
 * is the Start Of Frame signal (a single SE0 bit) available immediately after
 * a USB RESET. We first do a binary search for the OSCCAL value and then
 * optimize this value with a neighboorhod search.
 * This algorithm may also be used to calibrate the RC oscillator directly to
 * 12 MHz (no PLL involved, can therefore be used on almost ALL AVRs), but this
 * is wide outside the spec for the OSCCAL value and the required precision for
 * the 12 MHz clock! Use the RC oscillator calibrated to 12 MHz for
 * experimental purposes only!
 */
static void calibrateOscillator()
{
	uchar step = 128;
	uchar trialValue = 0, optimumValue;
	int x, optimumDev, targetValue = (unsigned)(1499 * (double)F_CPU / 10.5e6 + 0.5);

    /* do a binary search: */
    do
	{
        OSCCAL = trialValue + step;
        x = usbMeasureFrameLength();    /* proportional to current real frequency */
        if(x < targetValue)             /* frequency still too low */
            trialValue += step;
        step >>= 1;
    }
	while(step > 0);
    /* We have a precision of +/- 1 for optimum OSCCAL here */
    /* now do a neighborhood search for optimum value */
    optimumValue = trialValue;
    optimumDev = x; /* this is certainly far away from optimum */
    for(OSCCAL = trialValue - 1; OSCCAL <= trialValue + 1; OSCCAL++)
	{
        x = usbMeasureFrameLength() - targetValue;
        if(x < 0)
            x = -x;
        if(x < optimumDev)
		{
            optimumDev = x;
            optimumValue = OSCCAL;
        }
    }
    OSCCAL = optimumValue;
}
/*
Note: This calibration algorithm may try OSCCAL values of up to 192 even if
the optimum value is far below 192. It may therefore exceed the allowed clock
frequency of the CPU in low voltage designs!
You may replace this search algorithm with any other algorithm you like if
you have additional constraints such as a maximum CPU clock.
For version 5.x RC oscillators (those with a split range of 2x128 steps, e.g.
ATTiny25, ATTiny45, ATTiny85), it may be useful to search for the optimum in
both regions.
*/

void usbEventResetReady()
{
	cli();
	calibrateOscillator();
	sei();
	eeprom_update_byte((byte*)&eepCalibration, OSCCAL);   /* store the calibrated value in EEPROM */
}

void loadUSBOscCalibration()
{
	byte calibrationValue;
	calibrationValue = eeprom_read_byte(&eepCalibration); /* calibration value from last time */
	if(calibrationValue != 0xff)
		OSCCAL = calibrationValue;
}
