/*
 * Project: AVR ATtiny CPU Usage LEDs
 * Author: Zak Kemble, me@zakkemble.co.uk
 * Copyright: (C) 2012 by Zak Kemble
 * License: GNU GPL v3 (see License.txt)
 */

#define _BSD_SOURCE
#include "typedefs.h"
#include "backend.h"
#include "device.h"
#include <stdio.h>
#include <string.h>

#define SLEEPTIME 500

//static int hMainThread;

static void mainThread();
static byte getCPUUsage();
static bool checkDevice(int);

void backend()
{
//	pthread_create(&hMainThread, NULL, &mainThread, NULL);
	mainThread();
}

void backend_close()
{
//	pthread_join(hMainThread, NULL);
	device_close();
}

void backend_wait()
{

}

// Thread that gets and sends CPU usage
static void mainThread()
{
	int nBytes = 0;
	byte cpuUsage;

	while(1)
	{
		usleep(SLEEPTIME * 1000);

		// Get usage
		cpuUsage = getCPUUsage();

		// Show CPU usage
//		printf("Usage: %u%%\n", (byte)((cpuUsage / 255.0) * 100));

		// Send to USB
		if(checkDevice(nBytes))
			nBytes = device_sendCPU(cpuUsage);
	}
}

// Get CPU usage and return as a value from 0 to 255
static byte getCPUUsage()
{
	static long long last_idleTime		= 0;
	static long long last_kernelTime	= 0;
	static long long last_userTime		= 0;

	long long idleTime;
	long long kernelTime;
	long long userTime;

	// http://stackoverflow.com/questions/3017162/how-to-get-total-cpu-usage-in-linux-c
	
	FILE *fp = fopen("/proc/stat","r");
	char line[128];
	if(fp != NULL && fgets(line, 128, fp) != NULL)
	{
//		fputs(line, stdout);
		char* token = strtok(line, " ");
		int count = 0;
		while(token != NULL)
		{
//			printf("%s\n", token);
			
			if(count == 1)
				userTime = atoi(token);
			else if(count == 3)
				kernelTime = atoi(token);
			else if(count == 4)
				idleTime =  atoi(token);

			token = strtok(NULL, " ");
			count++;
		}
	}
	else
		perror("File open/read error");
	fclose(fp);

	long long usr = userTime - last_userTime;
	long long ker = kernelTime - last_kernelTime;
	long long idl = idleTime - last_idleTime;
	long long sys = ker + usr;

	last_idleTime = idleTime;
	last_kernelTime = kernelTime;
	last_userTime = userTime;

	if(sys == 0)
		return 0;

	return (byte)(((float)sys / (sys + idl)) * 255);
}

// See of the device is ok and connected, if not, try reconnecting
static bool checkDevice(int nBytes)
{
	if(nBytes < 0 || !device_valid())
	{
		device_close();
		device_open();

		if(!device_valid())
			return false;
	}
	return true;
}
