/*
 * Project: AVR ATtiny CPU Usage LEDs
 * Author: Zak Kemble, me@zakkemble.co.uk
 * Copyright: (C) 2012 by Zak Kemble
 * License: GNU GPL v3 (see License.txt)
 */

#include "typedefs.h"
#include "actions.h"
#include "device.h"
#include "colour.h"
#include "resource.h"
#include <shellapi.h>
#include <commctrl.h>
#include <stdio.h>

static HWND* settingsWnd;
static NOTIFYICONDATA* nid;

static void setTrayIcon(bool);
static void updateTrackBar();
static uint getItemInt(int);
static bool settingsDlgIsOpen();

void actions(HWND* _settingsWnd, NOTIFYICONDATA* _nid)
{
	settingsWnd = _settingsWnd;
	nid = _nid;
}

// Put into single colour mode and set the colour
void setSingleColour()
{
	uint r = getItemInt(IDC_EDIT_SGL_RED);
	uint g = getItemInt(IDC_EDIT_SGL_GREEN);
	uint b = getItemInt(IDC_EDIT_SGL_BLUE);
	if(r > 255)
		r = 255;
	if(g > 255)
		g = 255;
	if(b > 255)
		b = 255;
	byte red = r;
	byte green = g;
	byte blue = b;

	setItemInt(IDC_EDIT_SGL_RED,	red);
	setItemInt(IDC_EDIT_SGL_GREEN,	green);
	setItemInt(IDC_EDIT_SGL_BLUE,	blue);

	device_setSingleColour(red, green, blue);
}

// Set brightness which also puts it into CPU usage mode
void setUsageBrightness()
{
	uint bright = getItemInt(IDC_EDIT_USAGE_BRT_MAX);
	if(bright > 255)
		bright = 255;
	byte brightness = bright;

	setItemInt(IDC_EDIT_USAGE_BRT_MAX, brightness);

	updateTrackBar();

	device_setBrightness(brightness);
}

// Set transition time for CPU usage mode
void setTransitionTime()
{
	uint time = getItemInt(IDC_EDIT_TRANS_TIME);
	if(time > 60000)
		time = 60000;

	// Due to the way the transition timing interval is worked out (byte value for interval time in milliseconds)
	// the total transition time is rounded down to the nearest multiple of 256,
	// so work out what the total transition time is actually going to be and show the user.
	time -= time % 256;

	setItemInt(IDC_EDIT_TRANS_TIME, time);

	// Need to work out the time interval between each PWM duty cycle change and send that
	time = (uint)((time / 256) + 0.5);

	device_setTransitionTime(time);
}

// Set idle time for CPU usage mode
void setIdleTime()
{
	uint time = getItemInt(IDC_EDIT_IDLE_TIME);
	if(time > 255)
		time = 255;

	setItemInt(IDC_EDIT_IDLE_TIME, time);

	device_setIdleTime(time);
}

// Update GUI with current device settings
void showCurrentSettings()
{
	if(settingsDlgIsOpen())
	{
		if(device_valid())
		{
			device_getState();

			s_device* device = device_get();
			setItemInt(IDC_EDIT_SGL_RED,	device->rbg.red);
			setItemInt(IDC_EDIT_SGL_GREEN,	device->rbg.green);
			setItemInt(IDC_EDIT_SGL_BLUE,	device->rbg.blue);

			setItemInt(IDC_EDIT_IDLE_TIME, device->settings.idleTime);
			setItemInt(IDC_EDIT_USAGE_BRT_MAX, device->settings.maxBrightness);

			updateTrackBar();

			uint transTime = device->settings.transitionTime * 256;
			setItemInt(IDC_EDIT_TRANS_TIME, transTime);
		}
	}
}

// Show connection status and firmware version
// showCurrentSettings() must be called before this function
void showConnectionStatus()
{
	bool valid = device_valid();
	setTrayIcon(valid);

	if(settingsDlgIsOpen())
	{
		if(valid)
		{
			s_device* device = device_get();
			SetDlgItemText(*settingsWnd, IDC_STATIC_USB_STATUS, "Connected");
			char buff[18];
			sprintf(buff,"Firmware: %u.%u", device->version[0], device->version[1]);
			SetDlgItemText(*settingsWnd, IDC_STATIC_USB_FIRMWARE, buff);

		}
		else
		{
			SetDlgItemText(*settingsWnd, IDC_STATIC_USB_STATUS, "Disconnected");
			SetDlgItemText(*settingsWnd, IDC_STATIC_USB_FIRMWARE, "Firmware: -.-");
		}
	}
}

// Update tray icon, this should probably be in the main window procedure instead
static void setTrayIcon(bool connected)
{
	if(connected)
		nid->hIcon = LoadIcon(GetModuleHandle(NULL), MAKEINTRESOURCE(IDI_SMALL_ON));
	else
		nid->hIcon = LoadIcon(GetModuleHandle(NULL), MAKEINTRESOURCE(IDI_SMALL_OFF));
	Shell_NotifyIcon(NIM_MODIFY, nid);
}

// Show CPU load
void showCPULoad(byte cpuUsage)
{
	if(settingsDlgIsOpen())
	{
		char buff[9];
		sprintf(buff,"CPU: %0u", (byte)((100 / 255.0) * cpuUsage));
		SetDlgItemText(*settingsWnd, IDC_STATIC_CPU_USAGE, buff);
	}
}

// Update brightness track bar position
static void updateTrackBar()
{
	if(settingsDlgIsOpen())
		SendMessage(*settingsWnd, WM_UPDATE_TRACKBAR, 0, 0);
}

// Get int value of a control
static uint getItemInt(int item)
{
	return GetDlgItemInt(*settingsWnd, item, NULL, FALSE);
}

// Set a control value
void setItemInt(int item, uint num)
{
	char buff[12];
	sprintf(buff, "%d", num);
	SetDlgItemText(*settingsWnd, item, buff);
}

// Is the settings dialog open
static bool settingsDlgIsOpen()
{
	return (settingsWnd != NULL);
}

// Track bar position to hue colour
void getColourHue(short position)
{
	byte red;
	byte green;
	byte blue;
	getHue(position, &red, &green, &blue);
	setItemInt(IDC_EDIT_SGL_RED, red);
	setItemInt(IDC_EDIT_SGL_GREEN, green);
	setItemInt(IDC_EDIT_SGL_BLUE, blue);
}
