/*
 * Project: AVR ATtiny CPU Usage LEDs
 * Author: Zak Kemble, me@zakkemble.co.uk
 * Copyright: (C) 2012 by Zak Kemble
 * License: GNU GPL v3 (see License.txt)
 */

#include "win.h"
#include "typedefs.h"
#include "backend.h"
#include "device.h"
#include "actions.h"
#include "resource.h"

#define SLEEPTIME 500

static HANDLE hThreadExitEvent;
static HANDLE hMainThread;

static DWORD WINAPI mainThread(LPVOID args);
static byte getCPUUsage();
static bool checkDevice(int);

void backend()
{
	hThreadExitEvent	= CreateEvent(NULL, TRUE, FALSE, NULL);
	hMainThread			= CreateThread(NULL, 0, mainThread, NULL, 0, NULL);
}

void backend_close()
{
	SetEvent(hThreadExitEvent);
	WaitForSingleObject(hMainThread, 3000);
	CloseHandle(hMainThread);
	CloseHandle(hThreadExitEvent);
	device_close();
}

void backend_wait()
{
	WaitForSingleObject(hMainThread, INFINITE);
}

// Thread that gets and sends CPU usage
static DWORD WINAPI mainThread(LPVOID args)
{
	int nBytes = 0;
	byte cpuUsage;

	while(1)
	{
		// If exit event is set then exit thread, otherwise wait for timeout and do the usual CPU stuff
		if(WaitForSingleObject(hThreadExitEvent, SLEEPTIME) != WAIT_TIMEOUT)
			break;

		// Get usage
		cpuUsage = getCPUUsage();

		// Show CPU usage
		showCPULoad(cpuUsage);

		// Send to USB
		if(checkDevice(nBytes))
			nBytes = device_sendCPU(cpuUsage);
	}

	return EXIT_SUCCESS;
}

// Get CPU usage and return as a value from 0 to 255
static byte getCPUUsage()
{
	static long long last_idleTime		= 0;
	static long long last_kernelTime	= 0;
	static long long last_userTime		= 0;

	long long idleTime;
	long long kernelTime;
	long long userTime;
	bool res = GetSystemTimes((LPFILETIME)&idleTime, (LPFILETIME)&kernelTime, (LPFILETIME)&userTime);
	UNREFERENCED_PARAMETER(res);

	long long usr = userTime - last_userTime;
	long long ker = kernelTime - last_kernelTime;
	long long idl = idleTime - last_idleTime;
	long long sys = ker + usr;

	last_idleTime = idleTime;
	last_kernelTime = kernelTime;
	last_userTime = userTime;

	if(sys == 0)
		return 0;

	return (byte)((sys - idl) * 255 / sys);
}

// See of the device is ok and connected, if not, try reconnecting
static bool checkDevice(int nBytes)
{
	if(nBytes < 0 || !device_valid())
	{
		device_close();
		device_open();

		if(!device_valid())
		{
			showConnectionStatus();
			return false;
		}
		else
		{
			showCurrentSettings();
			showConnectionStatus();
		}
	}
	return true;
}
