/*
 * Project: AVR ATtiny CPU Usage LEDs
 * Author: Zak Kemble, me@zakkemble.co.uk
 * Copyright: (C) 2012 by Zak Kemble
 * License: GNU GPL v3 (see License.txt)
 */

#include "typedefs.h"
#include "device.h"
#include "mcu.h"
#include <usb.h>
#include <stdio.h>
#include <string.h>

#define USB_TIMEOUT 		2000

static s_device device;

static int device_send(byte, uint, uint, char*, uint);
static usb_dev_handle* usbOpenDevice(ushort, char*, ushort, char*);
static int usbGetDescriptorString(usb_dev_handle*, int, ushort, char*, uint);

void device_open()
{
	device.handle = usbOpenDevice(VEN_ID, VEN_NAME, DEV_ID, DEV_NAME);
}

void device_close()
{
	if(device_valid())
	{
		usb_close(device.handle);
		device.handle = NULL;
	}
}

bool device_valid()
{
	return (device.handle != NULL);
}

int device_sendCPU(byte value)
{
	return device_send(USB_REQ_CPUUSAGE, value, 0, NULL, 0);
}

void device_setBrightness(byte value)
{
	uint data = (value<<8) | SET_MODE_CPUUSAGE;
	device_send(USB_REQ_SETTING, data, 0, NULL, 0);
}

void device_setSingleColour(byte red, byte green, byte blue)
{
	uint data1 = (red<<8) | SET_MODE_SINGLECOL;
	uint data2 = (blue<<8) | green;
	device_send(USB_REQ_SETTING, data1, data2, NULL, 0);
}

void device_setTransitionTime(byte value)
{
	uint data = (value<<8) | SET_TRANSITION_TIME;
	device_send(USB_REQ_SETTING, data, 0, NULL, 0);
}

void device_setIdleTime(byte value)
{
	uint data = (value<<8) | SET_IDLE_TIME;
	device_send(USB_REQ_SETTING, data, 0, NULL, 0);
}

// Get firmware version and settings
void device_getState()
{
	char buffer[9];
	if(device_send(USB_REQ_DEVSTATE, 0, 0, buffer, sizeof(buffer)) > 0)
	{
		device.version[0]				= buffer[1];
		device.version[1]				= buffer[0];
		device.settings.ledMode			= buffer[2];
		device.settings.maxBrightness	= buffer[3];
		device.settings.transitionTime	= buffer[4];
		device.settings.idleTime		= buffer[5];
		device.rbg.red					= buffer[6];
		device.rbg.green				= buffer[7];
		device.rbg.blue					= buffer[8];
	}
}

static int device_send(byte reqType, uint data1, uint data2, char* out, uint size)
{
	if(!device_valid())
		return -1;

	return usb_control_msg(device.handle,
		USB_TYPE_VENDOR | USB_RECIP_DEVICE | USB_ENDPOINT_IN,
		reqType, data1, data2, out, size, USB_TIMEOUT);
}

s_device* device_get()
{
	return &device;
}





/*
 * Project: AVR ATtiny USB Tutorial at http://codeandlife.com/
 * Author: Joonas Pihlajamaa, joonas.pihlajamaa@iki.fi
 * Based on V-USB example code by Christian Starkjohann
 * Copyright: (C) 2012 by Joonas Pihlajamaa
 * License: GNU GPL v3 (see License.txt)
 */

static usb_dev_handle* usbOpenDevice(ushort vendor, char* vendorName, ushort product, char* productName)
{
	struct usb_bus *bus;
	struct usb_device *dev;
	char devVendor[256], devProduct[256];

	usb_dev_handle* _handle = NULL;

	usb_init();
	usb_find_busses();
	usb_find_devices();

	for(bus=usb_get_busses(); bus; bus=bus->next)
	{
		for(dev=bus->devices; dev; dev=dev->next)
		{
			if(dev->descriptor.idVendor != vendor || dev->descriptor.idProduct != product)
				continue;

			// we need to open the device in order to query strings
			if(!(_handle = usb_open(dev)))
			{
				fprintf(stderr, "Warning: cannot open USB device: %s\n", usb_strerror());
				continue;
			}

			// get vendor name
			if(usbGetDescriptorString(_handle, dev->descriptor.iManufacturer, 0x0409, devVendor, sizeof(devVendor)) < 0)
			{
				fprintf(stderr, "Warning: cannot query manufacturer for device: %s\n", usb_strerror());
				usb_close(_handle);
				continue;
			}

			// get product name
			if(usbGetDescriptorString(_handle, dev->descriptor.iProduct, 0x0409, devProduct, sizeof(devVendor)) < 0)
			{
				fprintf(stderr, "Warning: cannot query product for device: %s\n", usb_strerror());
				usb_close(_handle);
				continue;
			}

			// see if vendor name and product name are what we're after
			if(strcmp(devVendor, vendorName) == 0 && strcmp(devProduct, productName) == 0)
				return _handle;

			usb_close(_handle);
		}
	}

	// Not found
	return NULL;
}

// used to get descriptor strings for device identification
static int usbGetDescriptorString(usb_dev_handle *dev, int index, ushort langid, char *buf, uint buflen)
{
    char buffer[256];
    int rval, i;

	// make standard request GET_DESCRIPTOR, type string and given index
    // (e.g. dev->iProduct)
	rval = usb_control_msg(dev,
        USB_TYPE_STANDARD | USB_RECIP_DEVICE | USB_ENDPOINT_IN,
        USB_REQ_GET_DESCRIPTOR, (USB_DT_STRING << 8) + index, langid,
        buffer, sizeof(buffer), 1000);

    if(rval < 0) // error
		return rval;

    // rval should be bytes read, but buffer[0] contains the actual response size
	if((unsigned char)buffer[0] < rval)
		rval = (unsigned char)buffer[0]; // string is shorter than bytes read

	if(buffer[1] != USB_DT_STRING) // second byte is the data type
		return 0; // invalid return type

	// we're dealing with UTF-16LE here so actual chars is half of rval,
	// and index 0 doesn't count
	rval /= 2;

	// lossy conversion to ISO Latin1
	for(i = 1; i < rval && i < buflen; i++)
	{
		if(buffer[2 * i + 1] == 0)
			buf[i-1] = buffer[2 * i];
		else
			buf[i-1] = '?'; // outside of ISO Latin1 range
	}
	buf[i-1] = 0;

	return i-1;
}
