/*
 * Project: AVR ATtiny Fan controller
 * Author: Zak Kemble, me@zakkemble.co.uk
 * Copyright: (C) 2012 by Zak Kemble
 * License: GNU GPL v3 (see License.txt)
 */

#include "fanController.h"

int main(void)
{
#if CFG_WDT_DEBUG
	bool reset = (MCUSR & RESET_MASK);
#endif

#if INPUT_MODE == MODE_DIGITAL && CFG_WDT_ENABLE
	resetWDT = false;
#endif

	setupHardware();
	FAN_OFF();

	// Wait for power
	delayWDT(START_PWR_DELAY);

#if CFG_WDT_DEBUG
	// Error reset happened
	if(reset)
		resetError();
#endif

	// Max out
	FAN_ON();
	delayWDT(START_MAX_DELAY);

	// Load speed
#if INPUT_MODE == MODE_DIGITAL
	speed = eeprom_read_byte(&eepSpeed);
#endif

	sei();

#if INPUT_MODE == MODE_DIGITAL
	saveOfvCount	= 0;
	saveDelay		= false;
	bool singlePressWait;
	bool upButtonPressed;
	bool downButtonPressed;
	bool upButtonWasPressed;
	bool downButtonWasPressed;
#endif

	// Do main stuff
	setSpeed();
	setSleepMode(true);
	while(1)
	{
#if INPUT_MODE == MODE_DIGITAL
		// Speed save delay
		if(saveOfvCount > 250)
		{
			saveOfvCount = 0;
			saveDelay = false;
			TIMSK &= ~(1<<TOIE0);

			setSleepMode(true);

			// Save data
			eeprom_update_byte(&eepSpeed,speed);
		}

		upButtonWasPressed = false;
		downButtonWasPressed = false;
		
		// Loop here while buttons are pressed
		while(1)
		{
			upButtonPressed = speedUp();
			downButtonPressed = speedDown();

			// No buttons pressed
			if(!upButtonPressed && !downButtonPressed)
				break;

			// Neither button was pressed, but now there is, do long wait
			singlePressWait = (!upButtonWasPressed && !downButtonWasPressed);

			if(upButtonPressed && downButtonPressed && upButtonWasPressed != downButtonWasPressed) // Both buttons now pressed
			{
				if(upButtonPressed && !downButtonWasPressed) // Up pressed then down pressed = go straight to max
					speed = SPEED_MAX;
				else if(downButtonPressed && !upButtonWasPressed) // Down pressed then up pressed = go straight to min
					speed = SPEED_MIN;
			}
			upButtonWasPressed = upButtonPressed;
			downButtonWasPressed = downButtonPressed;

			setSpeed();
			setSaveDelay();

			// Ideally there shouldn't be any delays in the main loop
			if(singlePressWait)
				delay(DELAY_SGL_PRESS);	
			else
				delay(DELAY_PRESS);

			#if CFG_WDT_ENABLE
				if(resetWDT)
				{
					wdt_int_reset();
					resetWDT = false;
				}
			#endif
		}
#else
		ATOMIC_BLOCK(ATOMIC_FORCEON) // Don't want ADC ISR to change speed value while setting speed
		{
			setSpeed();
		}
#endif
		#if CFG_WDT_ENABLE
			wdt_int_reset();
			#if INPUT_MODE == MODE_DIGITAL
				resetWDT = false;
			#endif
		#endif

		// Sleep mode to save power and wait for an interrupt
		sleep_mode();
	}
}

void inline setupHardware()
{
#if CFG_WDT_ENABLE
	// Set watchdog
	wdt_enable(WDTO_2S);
	MCUSR = 0x00;
	wdt_int_reset();
#endif

	// Set frequency and power
	clock_prescale_set(CPU_DIV);
	power_usi_disable();
#if INPUT_MODE == MODE_DIGITAL
	power_adc_disable();
#endif
	power_timer1_disable();
	set_sleep_mode(SLEEP_MODE_IDLE);
	
	// Configure IO pins
	FAN_DDR |= (1<<FAN_PIN);
#if INPUT_MODE == MODE_DIGITAL
	// Pull-up on input switch pins
	SWITCHUP_PORT |= (1<<SWITCHUP_PIN);
	SWITCHDOWN_PORT |= (1<<SWITCHDOWN_PIN);
#endif

	// Timer settings
	TCCR0A = (1<<WGM00); // Phase correct PWM
	TCCR0B = (1<<CS00); // No prescale @ 31.25KHz = ~62Hz PWM

#if INPUT_MODE == MODE_DIGITAL
	// Interrupts on switch pins
	PCMSK = (1<<SWITCHUP_INT)|(1<<SWITCHDOWN_INT);
	GIMSK = (1<<PCIE);
#else
	// Setup ADC
	ADMUX = (1<<ADLAR)|(1<<MUX0);
	ADCSRA = (1<<ADEN)|(1<<ADATE)|(1<<ADIE)|(1<<ADPS2)|(1<<ADPS1)|(1<<ADPS0);
	DIDR0 = (1<<ADC1D);
#endif
}

#if INPUT_MODE == MODE_DIGITAL
bool inline speedUp()
{
	if(SWITCHUP_PRESSED())
	{
		if(speed + SPEED_STEP < SPEED_MAX)
			speed += SPEED_STEP;
		else
			speed = SPEED_MAX;
		return true;
	}
	return false;
}

bool inline speedDown()
{
	if(SWITCHDOWN_PRESSED())
	{
		if(speed - SPEED_STEP > SPEED_MIN)
			speed -= SPEED_STEP;
		else
			speed = SPEED_MIN;
		return true;
	}
	return false;
}
#endif

void setSpeed()
{
#if !CFG_SPEED_SWAP
	if(speed == 0) // Disconnect port from timer and set pin low.
		FAN_OFF();
	else if(speed == 255) // Disconnect port from timer and set pin high.
		FAN_ON();
	else
		FAN_SET(speed); // Enable timer, connect port to timer and set PWM value.
#else
	if(speed == 0)
		FAN_ON();
	else if(speed == 255)
		FAN_OFF();
	else
		FAN_SET(~speed);
#endif
	setSleepMode(false);
}

void setSleepMode(bool eepSaved)
{
#if INPUT_MODE == MODE_DIGITAL
	// If the speed has been saved and speed is either maximum or minimum we can turn off timers
	// and put the uC into the most power saving sleep mode, Power-down.
	// Otherwise it must be put into idle sleep mode so the timers stay active.
	if(eepSaved && (speed == 0 || speed == 255))
	{
		power_timer0_disable();
		set_sleep_mode(SLEEP_MODE_PWR_DOWN);
	}
	else
		set_sleep_mode(SLEEP_MODE_IDLE);
#else
	// The ADC is constantly in use so the uC can't be put into power-down sleep mode, but can be
	// put into ADC noise reduction sleep mode.
	// If timers are needed then idle sleep mode is used.
	if(speed == 0 || speed == 255)
	{
		power_timer0_disable();
		set_sleep_mode(SLEEP_MODE_ADC);
	}	
	else
		set_sleep_mode(SLEEP_MODE_IDLE);
#endif
}

#if CFG_WDT_ENABLE
void delayWDT_f(unsigned int ms)
{
	while(ms--)
	{
		delay(1);
		wdt_int_reset();
	}
}
#endif

#if CFG_WDT_DEBUG
void resetError()
{
	while(1)
	{
		delayWDT(2000);
		FAN_ON();
		delayWDT(500);
		FAN_OFF();
	}
}
#endif

#if INPUT_MODE == MODE_DIGITAL
ISR(PCINT0_vect)
{
// A button was pressed.
// Do nothing here, just wake CPU and let the main loop deal with it.
}

ISR(TIMER0_OVF_vect)
{
// This stuff is for saving the speed to EEPROM. This creates a ~4 second delay which is reset
// each time a button is pressed so EEPROM is not written every time, increasing the EEPROM life.

	if(saveDelay && saveOfvCount < 255) // Make sure saveOfvCount doesn't overflow if a long delay happens somewhere
		++saveOfvCount;
}
#else
ISR(ADC_vect)
{
// Only adjust speed if there is a difference of at least 5 between the analogue reading and the current speed.
	byte adc = ADCH;
	int tmp = adc - speed;
	if(tmp > 5 || tmp < 5 || adc <= 5 || adc >= 250)
		speed = adc;
}
#endif

ISR(WDT_vect)
{
#if INPUT_MODE == MODE_DIGITAL && CFG_WDT_ENABLE
	// Do nothing, just wake CPU so it can set WDT interrupt back on
	resetWDT = true;
#endif
}
