/*
 * Project: AVR ATtiny Fan controller
 * Author: Zak Kemble, me@zakkemble.co.uk
 * Copyright: (C) 2012 by Zak Kemble
 * License: GNU GPL v3 (see License.txt)
 */

#ifndef FANCONTROLLER_H_
#define FANCONTROLLER_H_

#include "util.h"
#include "config.h"
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/power.h>
#include <avr/sleep.h>
#include <avr/wdt.h>
#include <util/delay.h>
#if CFG_INPUT == MODE_DIGITAL
	#include <avr/eeprom.h>
#else
	#include <util/atomic.h>
#endif

#define RESET_MASK			0b00001000
#define DELAY_PRESS			CFG_SPEED_DELAY
#define DELAY_SGL_PRESS		CFG_SPEED_DELAY_SGL
#define START_PWR_DELAY		CFG_START_PWR_DELAY
#define START_MAX_DELAY		CFG_START_MAX_DELAY
#define SPEED_STEP			CFG_SPEED_STEP
#define SPEED_MAX			CFG_SPEED_MAX
#define SPEED_MIN			CFG_SPEED_MIN
#define FAN_PORT			PORT(CFG_FAN_PORT)
#define FAN_PIN				PORTBIT(CFG_FAN_PORT, CFG_FAN_PIN)
#define FAN_PWM				CFG_FAN_PWM
#define FAN_DDR				DDR(CFG_FAN_PORT)
#define SWITCHUP_INT		PCINT(CFG_SWITCHUP_PCINT)
#define SWITCHUP_PPORT		PINPORT(CFG_SWITCHUP_PORT)
#define SWITCHUP_PIN		PINBIT(CFG_SWITCHUP_PORT, CFG_SWITCHUP_PIN)
#define SWITCHUP_PORT		PORT(CFG_SWITCHUP_PORT)
#define SWITCHDOWN_INT		PCINT(CFG_SWITCHDOWN_PCINT)
#define SWITCHDOWN_PPORT	PINPORT(CFG_SWITCHDOWN_PORT)
#define SWITCHDOWN_PIN		PINBIT(CFG_SWITCHDOWN_PORT, CFG_SWITCHDOWN_PIN)
#define SWITCHDOWN_PORT		PORT(CFG_SWITCHDOWN_PORT)
#define INPUT_MODE			CFG_INPUT

#define setSaveDelay()		(saveDelay = true, saveOfvCount = 0, power_timer0_enable(), TIMSK |= (1<<TOIE0))
#define FAN_ON()			(TCCR0A &= ~(1<<COM0B1), FAN_PORT |= (1<<FAN_PIN))
#define FAN_OFF()			(TCCR0A &= ~(1<<COM0B1), FAN_PORT &= ~(1<<FAN_PIN))
#define FAN_SET(toSpeed)	(power_timer0_enable(), TCCR0A |= (1<<COM0B1), FAN_PWM = toSpeed)
#define SWITCHUP_PRESSED()	(!(SWITCHUP_PPORT & (1<<SWITCHUP_PIN)))
#define SWITCHDOWN_PRESSED()	(!(SWITCHDOWN_PPORT & (1<<SWITCHDOWN_PIN)))
#define wdt_int_reset()		(WDTCR |= (1<<WDIE)|(1<<WDE))
#if CFG_WDT_ENABLE
	#define delayWDT(ms)		(delayWDT_f(ms))
#else
	#define delayWDT(ms)		(delay(ms))
#endif

#if INPUT_MODE == MODE_ANALOGUE
	volatile byte speed;
#else
	byte speed;
	byte EEMEM eepSpeed = 0xFF;
	volatile byte saveOfvCount;
	volatile bool saveDelay;
	#if CFG_WDT_ENABLE
		volatile bool resetWDT;
	#endif
#endif

void inline setupHardware();
#if INPUT_MODE == MODE_DIGITAL
	bool inline speedUp();
	bool inline speedDown();
#endif
void setSpeed();
void setSleepMode(bool eepSaved);
#if CFG_WDT_ENABLE
	void delayWDT_f(unsigned int ms);
#endif
#if CFG_WDT_DEBUG
	void resetError();
#endif

#endif /* FANCONTROLLER_H_ */