/*
 * Project: AVR ATtiny Fan controller
 * Author: Zak Kemble, me@zakkemble.co.uk
 * Copyright: (C) 2012 by Zak Kemble
 * License: GNU GPL v3 (see License.txt)
 */

// If you're using very low CPU clock rates 
// and you're programming with SPI/ISP you'll
// probably have to reduce the programmer clock rate,
// for USBASP with avrdude use -B 32

// Fuses for ATtiny25/45/85
// 8MHz internal RC 6CK/14CK + 64ms startup
// Serial program (SPI) enabled
// Brown-out at 2.7V
// High:		0xDD
// Low:			0xE2
// Extended:	0xFF

// Oh another thing, some chips/revisions have problems with writing/reading EEPROM at very low frequencies,
// you can check to see if yours is affected by looking in the errata section of the chips datasheet.
// In this program, EEPROM is used for storing fan speed in digital mode.

#ifndef CONFIG_H_
#define CONFIG_H_

///////////////////////////////////////
// Don't need touch these two
#define MODE_DIGITAL			0
#define MODE_ANALOGUE			1
///////////////////////////////////////

// uC Frequency
#define F_CPU					31250 // If you change the frequency don't forget to change the timer prescaler, EEPROM save delay and maybe ADC prescaler
#define CPU_DIV					clock_div_256 // 8MHz / 256 = 31.25KHz

// Input
// MODE_DIGITAL = switches
// MODE_ANALOGUE = potentiometer
#define CFG_INPUT				MODE_DIGITAL

// Watchdog
#define CFG_WDT_ENABLE			1 // If the controller crashes for some reason then automatically reset it
#define CFG_WDT_DEBUG			1 // Debug makes the fan turn on and off if the controller was reset by the watchdog

// Fan PWM output
#define CFG_FAN_PORT			B
#define CFG_FAN_PIN				1
#define CFG_FAN_PWM				OCR0B

// Digital mode
// Speed up and down switches
#define CFG_SWITCHUP_PORT		B
#define CFG_SWITCHUP_PIN		0
#define CFG_SWITCHUP_PCINT		0 // Interrupt for this pin, PCINT0 etc
#define CFG_SWITCHDOWN_PORT		B
#define CFG_SWITCHDOWN_PIN		2
#define CFG_SWITCHDOWN_PCINT	2 // Interrupt for this pin, PCINT0 etc

// Digital mode
// How fast speed changes while holding a button
#define CFG_SPEED_STEP			16 // Step size for changing speed
#define CFG_SPEED_DELAY			150 // While a button is held down this is the delay for adding another speed step
#define CFG_SPEED_DELAY_SGL		300 // Delay for when a button is first pressed, makes it easier to just press the button once for a single change in speed

// Analogue mode
// If you change the ADC port/pin you'll also need to update ADMUX and DIDRn registers in setupHardware()
#define CFG_ANALOGUE_PORT		B
#define CFG_ANALOGUE_PIN		2

// Power on delay
#define CFG_START_PWR_DELAY		200

// Fan start boost time
#define CFG_START_MAX_DELAY		2000

// Minimum & maximum speed
#define CFG_SPEED_MAX			255
#define CFG_SPEED_MIN			0

// Depending on how you've wired up the potentiometer, you
// may want to swap which end is max speed and min speed.
#define CFG_SPEED_SWAP			0

#endif /* CONFIG_H_ */