/*
 * Project: AVR ATtiny Fan controller
 * Author: Zak Kemble, me@zakkemble.co.uk
 * Copyright: (C) 2012 by Zak Kemble
 * License: GNU GPL v3 (see License.txt)
 */

#include "fanController.h"

int main(void)
{
#if CFG_WDT_DEBUG
	bool reset = (MCUSR & RESET_MASK);
#endif

	setupHardware();
	fanSet(0);

	// Wait for power
	delayWDT(START_PWR_DELAY);

#if CFG_WDT_DEBUG
	// Error reset happened
	if(reset)
		resetError();
#endif

	// Max out
	fanSet(255);
	delayWDT(START_MAX_DELAY);

#if INPUT_MODE == MODE_DIGITAL
	// Load speed
	speed = eeprom_read_byte(&eepSpeed);

	#if CFG_WDT_ENABLE
		resetWDT = false;
	#endif

	saveOfvCount	= 0;
	saveDelay		= false;
	bool singlePressWait;
	bool upButtonPressed;
	bool downButtonPressed;
	bool upButtonWasPressed;
	bool downButtonWasPressed;

	// Set initial speed
	fanSet(speed);
	setSleepMode(true);
#endif

	sei();

	// Do main stuff
	while(1)
	{
#if INPUT_MODE == MODE_DIGITAL
		// Speed save delay
		if(saveOfvCount > 250)
		{
			saveOfvCount = 0;
			saveDelay = false;
			TIMSK &= ~(1<<TOIE0);

			setSleepMode(true);

			// Save data
			eeprom_update_byte(&eepSpeed,speed);
		}

		upButtonWasPressed = false;
		downButtonWasPressed = false;
		
		// Loop here while buttons are pressed
		while(1)
		{
			upButtonPressed = speedUp();
			downButtonPressed = speedDown();

			// No buttons pressed
			if(!upButtonPressed && !downButtonPressed)
				break;

			// Neither button was pressed, but now there is, do long wait
			singlePressWait = (!upButtonWasPressed && !downButtonWasPressed);

			if(upButtonPressed && downButtonPressed && upButtonWasPressed != downButtonWasPressed) // Both buttons now pressed
			{
				if(upButtonPressed && !downButtonWasPressed) // Up pressed then down pressed = go straight to max
					speed = SPEED_MAX;
				else if(downButtonPressed && !upButtonWasPressed) // Down pressed then up pressed = go straight to min
					speed = SPEED_MIN;
			}
			upButtonWasPressed = upButtonPressed;
			downButtonWasPressed = downButtonPressed;

			fanSet(speed);
			setSleepMode(false);
			setSaveDelay();

			// Ideally there shouldn't be any delays in the main loop
			if(singlePressWait)
				delay(DELAY_SGL_PRESS);	
			else
				delay(DELAY_PRESS);

			#if CFG_WDT_ENABLE
				if(resetWDT)
				{
					wdt_int_reset();
					resetWDT = false;
				}
			#endif
		}
#else
		ATOMIC_BLOCK(ATOMIC_FORCEON) // Don't want ADC ISR to change speed value while setting speed
		{
			fanSet(speed);
			setSleepMode();
		}
#endif
		#if CFG_WDT_ENABLE
			wdt_int_reset();
			#if INPUT_MODE == MODE_DIGITAL
				resetWDT = false;
			#endif
		#endif

		// Sleep mode to save power and wait for an interrupt
		sleep_mode();
	}
}

static void setupHardware()
{
#if CFG_WDT_ENABLE
	// Set watchdog
	wdt_enable(WDTO_2S);
	MCUSR = 0x00;
	wdt_int_reset();
#endif

	// Set frequency and power
	clock_prescale_set(CPU_DIV);
	power_usi_disable();
#if INPUT_MODE == MODE_DIGITAL
	power_adc_disable();
#endif
	power_timer1_disable();
	set_sleep_mode(SLEEP_MODE_IDLE);

	// Timer settings
	TCCR0A = (1<<WGM00); // Phase correct PWM
	TCCR0B = (1<<CS00); // No prescale @ 31.25KHz = ~62Hz PWM

	// Configure IO pins
	FAN_DDR |= (1<<FAN_PIN); // Fan pin as output

#if INPUT_MODE == MODE_DIGITAL
	// Pull-up on all pins apart from fan pin
	PORTB = (0xFF ^ (1<<FAN_PIN));

	// Interrupts on switch pins
	PCMSK = (1<<SWITCHUP_INT)|(1<<SWITCHDOWN_INT);
	GIMSK = (1<<PCIE);
#else
	// Pull-up on all pins apart from fan pin and analogue input pin
	PORTB = (0xFF ^ ((1<<FAN_PIN)|(1<<ANALOGUE_PIN)));

	// Setup ADC
	ADMUX = (1<<ADLAR)|(1<<MUX0);
	ADCSRA = (1<<ADEN)|(1<<ADATE)|(1<<ADIE)|(1<<ADPS2)|(1<<ADPS1)|(1<<ADPS0);
	DIDR0 = (1<<ADC1D);
#endif
}

#if INPUT_MODE == MODE_DIGITAL
// If speed up button pressed, increase speed
static bool speedUp()
{
	if(SWITCHUP_PRESSED())
	{
		if(speed + SPEED_STEP < SPEED_MAX)
			speed += SPEED_STEP;
		else
			speed = SPEED_MAX;
		return true;
	}
	return false;
}

// If slow down button pressed, decrease speed
static bool speedDown()
{
	if(SWITCHDOWN_PRESSED())
	{
		if(speed - SPEED_STEP > SPEED_MIN)
			speed -= SPEED_STEP;
		else
			speed = SPEED_MIN;
		return true;
	}
	return false;
}

// Begin timing for EEPROM speed save delay
static void setSaveDelay()
{
	saveOfvCount = 0;
	saveDelay = true;
	power_timer0_enable();
	TIMSK |= (1<<TOIE0);
}

static void setSleepMode(bool eepSaved)
{
// If the speed has been saved and speed is either maximum or minimum we can turn off timers and put the uC into the most power saving sleep mode, Power-down.
// Otherwise it must be put into idle sleep mode so the timers stay active.
	if(eepSaved && (speed == 0 || speed == 255))
	{
		power_timer0_disable();
		set_sleep_mode(SLEEP_MODE_PWR_DOWN);
	}
	else
		set_sleep_mode(SLEEP_MODE_IDLE);
}
#else
static void setSleepMode()
{
// The ADC is constantly in use so the uC can't be put into power-down sleep mode, but can be put into ADC noise reduction sleep mode.
// If timers are needed then idle sleep mode is used.
	if(speed == 0 || speed == 255)
		set_sleep_mode(SLEEP_MODE_ADC);
	else
		set_sleep_mode(SLEEP_MODE_IDLE);
}

// This is the map function from Arduino, modified for byte values instead of long
static byte map(byte x, byte in_min, byte in_max, byte out_min, byte out_max)
{
// At least one of the values must be 32 bit, in this case x is type casted to long since some parts of the calculation can return values down to -65025.
// Or something like that... without the type cast the return value isn't quite right, map(255,0,255,0,255) would return 254

	return (((long)x - in_min) * (out_max - out_min) / (in_max - in_min) + out_min);
}
#endif

// Set fan speed to specified value
static void fanSet(byte speed)
{
	if(speed == 0 || speed == 255)
	{
		TCCR0A &= ~(1<<COM0B1);
		FAN_PORT = (FAN_PORT & ~(1<<FAN_PIN)) | ((!!speed)<<FAN_PIN);

// In digital mode Timer0 disable is done in setSleepMode() function since the EEPROM speed save timer uses it
#if INPUT_MODE == MODE_ANALOGUE
		power_timer0_disable();
#endif
	}
	else
	{
		power_timer0_enable();
		TCCR0A |= (1<<COM0B1);
		FAN_PWM = speed;
	}
}

#if CFG_WDT_ENABLE
// Delay and constantly reset watchdog timer
static void delayWDT_f(unsigned int ms)
{
	while(ms--)
	{
		delay(1);
		wdt_reset();
	}
}
#endif

#if CFG_WDT_DEBUG
// Turn fan on and off if the controller was restarted by the watchdog
static void resetError()
{
	while(1)
	{
		delayWDT(2000);
		fanSet(255);
		delayWDT(500);
		fanSet(0);
	}
}
#endif

#if INPUT_MODE == MODE_DIGITAL
ISR(PCINT0_vect)
{
// A button was pressed.
// Do nothing here, just wake CPU and let the main loop deal with it.
}

ISR(TIMER0_OVF_vect)
{
// This stuff is for saving the speed to EEPROM. This creates a ~4 second delay which is reset
// each time a button is pressed so EEPROM is not written every time, increasing the EEPROM life.

	if(saveDelay && saveOfvCount < 255) // Make sure saveOfvCount doesn't overflow if a long delay happens somewhere
		++saveOfvCount;
}
#else
ISR(ADC_vect)
{
#if !CFG_SPEED_SWAP
	speed = map(ADCH, 0, 255, SPEED_MIN, SPEED_MAX);
#else
	speed = map(ADCH, 0, 255, SPEED_MAX, SPEED_MIN);
#endif
}
#endif

ISR(WDT_vect)
{
#if INPUT_MODE == MODE_DIGITAL && CFG_WDT_ENABLE
	// Do nothing, just wake CPU so it can set WDT interrupt back on in the main loop.
	resetWDT = true;
#endif
}
