/*
 * Project: AVR ATtiny Fan controller
 * Author: Zak Kemble, contact@zakkemble.co.uk
 * Copyright: (C) 2013 by Zak Kemble
 * License: GNU GPL v3 (see License.txt)
 * Web: http://blog.zakkemble.co.uk/avr-microcontroller-based-pwm-fan-controllers/
 */

#include "config.h"
#include "util.h"
#include "typedefs.h"
#include "fanController.h"
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/power.h>
#include <avr/sleep.h>
#include <avr/wdt.h>
#include <util/delay.h>

#if INPUT_MODE == MODE_DIGITAL
	#include <avr/eeprom.h>
	static const byte EEMEM eepSpeed = SPEED_MAX;
	static byte speed;
	static volatile byte saveOfvCount;
	static volatile bool saveDelay;
	#if CFG_WDT_ENABLE
		static volatile bool resetWDT;
	#endif
	static void digitalStuff(void);
	static void doSaveDelay(void);
	static bool speedUp(void);
	static bool speedDown(void);
	static void setSaveDelay(void);
	static void setSleepMode(bool);
#elif INPUT_MODE == MODE_ANALOGUE
	static void setSleepMode(void);
	static volatile byte speed;
#endif

static void init(void);
static void fanSet(byte);
#if CFG_WDT_ENABLE
	static void delayWDT_f(uint);
#endif
#if CFG_WDT_DEBUG
	static void resetError(void);
#endif

int main(void)
{
#if CFG_WDT_DEBUG
	bool WDTreset = (MCUSR & (1<<WDRF));
#endif // CFG_WDT_DEBUG

	init();
	fanSet(0);

	// Wait for power
	delayWDT(START_PWR_DELAY);

#if CFG_WDT_DEBUG
	// Error reset happened
	if(WDTreset)
		resetError();
#endif // CFG_WDT_DEBUG

	// Max out
	fanSet(255);
	delayWDT(START_MAX_DELAY);

#if INPUT_MODE == MODE_DIGITAL
	// Load speed
	speed = eeprom_read_byte(&eepSpeed);

#if CFG_WDT_ENABLE
	resetWDT = false;
#endif // CFG_WDT_ENABLE

	saveOfvCount	= 0;
	saveDelay		= false;

	// Set initial speed
	fanSet(speed);
	setSleepMode(true);
#endif // INPUT_MODE

	sei();

	// Do main stuff
	while(1)
	{
#if INPUT_MODE == MODE_DIGITAL
		digitalStuff();
#elif INPUT_MODE == MODE_ANALOGUE
		fanSet(speed);
		setSleepMode();
#endif

#if CFG_WDT_ENABLE
		wdt_int_reset();
#if INPUT_MODE == MODE_DIGITAL
		resetWDT = false;
#endif // INPUT_MODE
#endif // CFG_WDT_ENABLE

		// Sleep mode to save power and wait for an interrupt
		sleep_mode();
	}
}

static void init()
{
#if CFG_WDT_ENABLE
	// Set watchdog
	wdt_enable(WDTO_2S);
	MCUSR = 0x00;
	wdt_int_reset();
#endif // CFG_WDT_ENABLE

	// Set frequency and power
	clock_prescale_set(CPU_DIV);
	power_usi_disable();
#if INPUT_MODE == MODE_DIGITAL
	power_adc_disable();
#endif // INPUT_MODE
	power_timer1_disable();
	set_sleep_mode(SLEEP_MODE_IDLE);

	// Timer settings
	TCCR0A = _BV(WGM00); // Phase correct PWM
	TCCR0B = _BV(CS00); // No prescale @ 31.25KHz = ~62Hz PWM

	// Configure IO pins
	FAN_DDR |= _BV(FAN_PIN); // Fan pin as output

#if INPUT_MODE == MODE_DIGITAL
	// Pull-up on all pins apart from fan pin
	PORTB = (0xFF ^ _BV(FAN_PIN));

	// Interrupts on switch pins
	PCMSK = _BV(SWITCHUP_INT)|_BV(SWITCHDOWN_INT);
	GIMSK = _BV(PCIE);
#elif INPUT_MODE == MODE_ANALOGUE
	// Pull-up on all pins apart from fan pin and analogue input pin
	PORTB = (0xFF ^ (_BV(FAN_PIN)|_BV(ANALOGUE_PIN)));

	// Setup ADC
	ADMUX = _BV(ADLAR)|_BV(MUX0);
	ADCSRA = _BV(ADEN)|_BV(ADATE)|_BV(ADIE)|_BV(ADPS2)|_BV(ADPS1)|_BV(ADPS0);
	DIDR0 = _BV(ADC1D);
#endif // INPUT_MODE
}

#if INPUT_MODE == MODE_DIGITAL
static void digitalStuff()
{
	doSaveDelay();

	bool upButtonWasPressed = false;
	bool downButtonWasPressed = false;

	// Loop here while buttons are pressed
	while(1)
	{
		bool upButtonPressed = speedUp();
		bool downButtonPressed = speedDown();

		// No buttons pressed
		if(!upButtonPressed && !downButtonPressed)
			break;

		// Neither button was pressed, but now there is, do long wait
		bool singlePressWait = (!upButtonWasPressed && !downButtonWasPressed);

		if(upButtonPressed && downButtonPressed && upButtonWasPressed != downButtonWasPressed) // Both buttons now pressed
		{
			if(upButtonPressed && !downButtonWasPressed) // Up pressed then down pressed = go straight to max
				speed = SPEED_MAX;
			else if(downButtonPressed && !upButtonWasPressed) // Down pressed then up pressed = go straight to min
				speed = SPEED_MIN;
		}
		upButtonWasPressed = upButtonPressed;
		downButtonWasPressed = downButtonPressed;

		fanSet(speed);
		setSleepMode(false);
		setSaveDelay();

		// Ideally there shouldn't be any delays
		singlePressWait ? delay(DELAY_SGL_PRESS) : delay(DELAY_PRESS);

#if CFG_WDT_ENABLE
		if(resetWDT)
		{
			wdt_int_reset();
			resetWDT = false;
		}
#endif // CFG_WDT_ENABLE
	}
}

static void doSaveDelay()
{
	// Speed save delay
	if(saveOfvCount > 250)
	{
		saveOfvCount = 0;
		saveDelay = false;

		// Turn off overflow interrupt
		TIMSK &= ~_BV(TOIE0);

		setSleepMode(true);

		// Save data
		eeprom_update_byte(&eepSpeed,speed);
	}
}

// If speed up button pressed, increase speed
static bool speedUp()
{
	if(SWITCHUP_PRESSED())
	{
		if(speed + SPEED_STEP < SPEED_MAX)
			speed += SPEED_STEP;
		else
			speed = SPEED_MAX;
		return true;
	}
	return false;
}

// If slow down button pressed, decrease speed
static bool speedDown()
{
	if(SWITCHDOWN_PRESSED())
	{
		if(speed - SPEED_STEP > SPEED_MIN)
			speed -= SPEED_STEP;
		else
			speed = SPEED_MIN;
		return true;
	}
	return false;
}

// Begin timing for EEPROM speed save delay
static void setSaveDelay()
{
	saveOfvCount = 0;
	saveDelay = true;

	// Make sure timer0 is on
	power_timer0_enable();

	// Enable overflow interrupt
	TIMSK |= _BV(TOIE0);
}

static void setSleepMode(bool eepSaved)
{
// If the speed has been saved and speed is either maximum or minimum we can turn off timers and put the uC into the most power saving sleep mode, Power-down.
// Otherwise it must be put into idle sleep mode so the timers stay active.
	if(eepSaved && (speed == 0 || speed == 255))
	{
		power_timer0_disable();
		set_sleep_mode(SLEEP_MODE_PWR_DOWN);
	}
	else
		set_sleep_mode(SLEEP_MODE_IDLE);
}
#elif INPUT_MODE == MODE_ANALOGUE
static void setSleepMode()
{
// The ADC is constantly in use so the uC can't be put into power-down sleep mode, but can be put into ADC noise reduction sleep mode.
// If timers are needed then idle sleep mode is used.
	if(speed == 0 || speed == 255)
		set_sleep_mode(SLEEP_MODE_ADC);
	else
		set_sleep_mode(SLEEP_MODE_IDLE);
}
#endif // INPUT_MODE

// Set fan speed to specified value
static void fanSet(byte speed)
{
	if(speed == 0 || speed == 255)
	{
		TCCR0A &= ~_BV(COM0B1);
		if(speed == 255)
			FAN_PORT |= _BV(FAN_PIN);
		else
			FAN_PORT &= ~_BV(FAN_PIN);

// In digital mode Timer0 disable is done in setSleepMode() function since the EEPROM speed save timer uses it
#if INPUT_MODE == MODE_ANALOGUE
		power_timer0_disable();
#endif // INPUT_MODE
	}
	else
	{
		power_timer0_enable();
		TCCR0A |= _BV(COM0B1);
		FAN_PWM = speed;
	}
}

#if CFG_WDT_ENABLE
// Delay while resetting watchdog timer
static void delayWDT_f(unsigned int ms)
{
	while(ms--)
	{
		delay(1);
		wdt_reset();
	}
}
#endif // CFG_WDT_ENABLE

#if CFG_WDT_DEBUG
// Turn fan on and off if the controller was restarted by the watchdog
static void resetError()
{
	while(1)
	{
		delayWDT(2000);
		fanSet(255);
		delayWDT(500);
		fanSet(0);
	}
}
#endif // CFG_WDT_DEBUG

#if INPUT_MODE == MODE_DIGITAL
ISR(PCINT0_vect)
{
// A button was pressed.
// Do nothing here, just wake CPU and let the main loop deal with it.
}

ISR(TIMER0_OVF_vect)
{
// This stuff is for saving the speed to EEPROM. This creates a ~4 second delay which is reset
// each time a button is pressed so EEPROM is not written every time, increasing the EEPROM life.

	if(saveDelay && saveOfvCount < 255) // Make sure saveOfvCount doesn't overflow if a long delay happens somewhere
		++saveOfvCount;
}
#elif INPUT_MODE == MODE_ANALOGUE
ISR(ADC_vect)
{
#if !CFG_SPEED_SWAP
	speed = map(ADCH, 0, 255, SPEED_MIN, SPEED_MAX);
#else
	speed = map(ADCH, 0, 255, SPEED_MAX, SPEED_MIN);
#endif // CFG_SPEED_SWAP
}
#endif // INPUT_MODE

ISR(WDT_vect)
{
#if INPUT_MODE == MODE_DIGITAL && CFG_WDT_ENABLE
	// Do nothing, just wake CPU so it can set WDT interrupt back on in the main loop.
	resetWDT = true;
#endif // INPUT_MODE
}
