/*
 * Project: AVR ATtiny Fan controller
 * Author: Zak Kemble, contact@zakkemble.co.uk
 * Copyright: (C) 2013 by Zak Kemble
 * License: GNU GPL v3 (see License.txt)
 * Web: http://blog.zakkemble.co.uk/avr-microcontroller-based-pwm-fan-controllers/
 */

#include "config.h"
#include "util.h"
#include "typedefs.h"
#include "fanController.h"
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/power.h>
#include <avr/sleep.h>
#include <avr/wdt.h>
#include <util/delay.h>

static void init(void);
static void fanSet(byte);
#if CFG_WDT_ENABLE
	static void delayWDT_f(uint);
#endif
#if CFG_WDT_DEBUG
	static void resetError(void);
#endif

static volatile byte speed;

int main(void)
{
#if CFG_WDT_DEBUG
	bool WDTreset = (RSTFLR & _BV(WDRF));
#endif

	init();
	fanSet(0);

	// Wait for power
	delayWDT(START_PWR_DELAY);

#if CFG_WDT_DEBUG
	// Error reset happened
	if(WDTreset)
		resetError();
#endif

	// Max out
	fanSet(255);
	delayWDT(START_MAX_DELAY);

	sei();

	// Do main stuff
	while(1)
	{
		fanSet(speed);

#if CFG_WDT_ENABLE
		wdt_int_reset();
#endif

		// Sleep mode to save power and wait for an interrupt
		sleep_mode();
	}
}

static void init(void)
{
#if CFG_WDT_ENABLE
	// Set watchdog
	wdt_enable(WDTO_2S);
	RSTFLR = 0x00;
	wdt_int_reset();
#endif

	// Set frequency and power
	clock_prescale_set(CPU_DIV);
	set_sleep_mode(SLEEP_MODE_IDLE);

	// Disable unused stuff
	ACSR = _BV(ACD);

	// Timer settings
	TCCR0A = _BV(WGM00); // Phase correct PWM
	TCCR0B = _BV(CS00); // No prescale @ 31.25KHz = ~62Hz PWM

	// Configure IO pins
	FAN_DDR |= _BV(FAN_PIN); // Fan pin as output

	// Pull-up on all pins apart from fan pin and analogue input pin
	PORTB = (0xFF ^ (_BV(FAN_PIN)|_BV(ANALOGUE_PIN)));

	// Setup ADC
	ADMUX = _BV(MUX0);
	ADCSRA = _BV(ADEN)|_BV(ADATE)|_BV(ADIE)|_BV(ADPS1)|_BV(ADPS0);
	ADCSRB = _BV(ADTS2);
	DIDR0 = _BV(ADC1D);
}

// Set fan speed
static void fanSet(byte speed)
{
	if(speed == 0 || speed == 255)
	{
		TCCR0A &= ~_BV(COM0A1);
		if(speed == 255)
			FAN_PORT |= _BV(FAN_PIN);
		else
			FAN_PORT &= ~_BV(FAN_PIN);
	}
	else
	{
		TCCR0A |= _BV(COM0A1);
		FAN_PWM = speed;
	}
}

#if CFG_WDT_ENABLE
// Delay while resetting watchdog timer
static void delayWDT_f(uint ms)
{
	while(ms--)
	{
		delay(1);
		wdt_reset();
	}
}
#endif

#if CFG_WDT_DEBUG
// Turn fan on and off if the controller was restarted by the watchdog
static void resetError(void)
{
	while(1)
	{
		delayWDT(2000);
		fanSet(255);
		delayWDT(500);
		fanSet(0);
	}
}
#endif

// ADC complete
ISR(ADC_vect)
{
#if CFG_SPEED_SWAP
	speed = ~ADCL;
#else
	speed = ADCL;
#endif
}

ISR(WDT_vect)
{
// Do nothing, just wake CPU so it can set WDT interrupt back on in the main loop.
}
